% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dplyr-between_time.R
\name{between_time}
\alias{between_time}
\title{Between (For Time Series): Range detection for date or date-time sequences}
\usage{
between_time(index, start_date = "start", end_date = "end")
}
\arguments{
\item{index}{A date or date-time vector.}

\item{start_date}{The starting date}

\item{end_date}{The ending date}
}
\value{
A \code{logical} vector the same length as \code{index} indicating whether or not
the timestamp value was within the \code{start_date} and \code{end_date} range.
}
\description{
The easiest way to filter time series date or date-time vectors. Returns a
logical vector indicating which date or date-time values are within a range.
See \code{\link[=filter_by_time]{filter_by_time()}} for the \code{data.frame} (\code{tibble}) implementation.
}
\details{
\strong{Pure Time Series Filtering Flexibilty}

The \code{start_date}  and \code{end_date} parameters are designed with flexibility in mind.

Each side of the \code{time_formula} is specified as the character
\code{'YYYY-MM-DD HH:MM:SS'}, but powerful shorthand is available.
Some examples are:
\itemize{
\item \strong{Year:} \verb{start_date = '2013', end_date = '2015'}
\item \strong{Month:} \verb{start_date = '2013-01', end_date = '2016-06'}
\item \strong{Day:} \verb{start_date = '2013-01-05', end_date = '2016-06-04'}
\item \strong{Second:} \verb{start_date = '2013-01-05 10:22:15', end_date = '2018-06-03 12:14:22'}
\item \strong{Variations:} \verb{start_date = '2013', end_date = '2016-06'}
}

\strong{Key Words: "start" and "end"}

Use the keywords "start" and "end" as shorthand, instead of specifying the
actual start and end values. Here are some examples:
\itemize{
\item \strong{Start of the series to end of 2015:} \verb{start_date = 'start', end_date = '2015'}
\item \strong{Start of 2014 to end of series:} \verb{start_date = '2014', end_date = 'end'}
}

\strong{Internal Calculations}

All shorthand dates are expanded:
\itemize{
\item The \code{start_date} is expanded to be the \emph{first date} in that period
\item The \code{end_date} side is expanded to be the \emph{last date} in that period
}

This means that the following examples are equivalent (assuming your
index is a POSIXct):
\itemize{
\item \code{start_date = '2015'} is equivalent to \code{start_date = '2015-01-01 + 00:00:00' }
\item \code{end_date = '2016'} is equivalent to \verb{2016-12-31 + 23:59:59'}
}
}
\examples{
library(dplyr)
library(timetk)

index_daily <- tk_make_timeseries("2016-01-01", "2017-01-01", by = "day")
index_min   <- tk_make_timeseries("2016-01-01", "2017-01-01", by = "min")

# How it works
# - Returns TRUE/FALSE length of index
# - Use sum() to tally the number of TRUE values
index_daily \%>\% between_time("start", "2016-01") \%>\% sum()

# ---- INDEX SLICING ----

# Daily Series: Month of January 2016
index_daily[index_daily \%>\% between_time("start", "2016-01")]

# Daily Series: March 1st - June 15th, 2016
index_daily[index_daily \%>\% between_time("2016-03", "2016-06-15")]

# Minute Series:
index_min[index_min \%>\% between_time("2016-02-01 12:00", "2016-02-01 13:00")]

# ---- FILTERING WITH DPLYR ----
FANG \%>\%
    group_by(symbol) \%>\%
    filter(date \%>\% between_time("2016-01", "2016-01"))

}
\references{
\itemize{
\item This function is based on the \code{tibbletime::filter_time()} function developed by Davis Vaughan.
}
}
\seealso{
Time-Based dplyr functions:
\itemize{
\item \code{\link[=summarise_by_time]{summarise_by_time()}} - Easily summarise using a date column.
\item \code{\link[=mutate_by_time]{mutate_by_time()}} - Simplifies applying mutations by time windows.
\item \code{\link[=pad_by_time]{pad_by_time()}} - Insert time series rows with regularly spaced timestamps
\item \code{\link[=filter_by_time]{filter_by_time()}} - Quickly filter using date ranges.
\item \code{\link[=filter_period]{filter_period()}} - Apply filtering expressions inside periods (windows)
\item \code{\link[=slice_period]{slice_period()}} - Apply slice inside periods (windows)
\item \code{\link[=condense_period]{condense_period()}} - Convert to a different periodicity
\item \code{\link[=between_time]{between_time()}} - Range detection for date or date-time sequences.
\item \code{\link[=slidify]{slidify()}} - Turn any function into a sliding (rolling) function
}
}
