% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/step_adasyn.R
\name{step_adasyn}
\alias{step_adasyn}
\title{Apply Adaptive Synthetic Algorithm}
\usage{
step_adasyn(
  recipe,
  ...,
  role = NA,
  trained = FALSE,
  column = NULL,
  over_ratio = 1,
  neighbors = 5,
  skip = TRUE,
  seed = sample.int(10^5, 1),
  id = rand_id("adasyn")
)
}
\arguments{
\item{recipe}{A recipe object. The step will be added to the
sequence of operations for this recipe.}

\item{...}{One or more selector functions to choose which
variable is used to sample the data. See \code{\link[=selections]{selections()}}
for more details. The selection should result in \emph{single
factor variable}. For the \code{tidy} method, these are not
currently used.}

\item{role}{Not used by this step since no new variables are
created.}

\item{trained}{A logical to indicate if the quantities for
preprocessing have been estimated.}

\item{column}{A character string of the variable name that will
be populated (eventually) by the \code{...} selectors.}

\item{over_ratio}{A numeric value for the ratio of the
majority-to-minority frequencies. The default value (1) means
that all other levels are sampled up to have the same
frequency as the most occurring level. A value of 0.5 would mean
that the minority levels will have (at most) (approximately)
half as many rows than the majority level.}

\item{neighbors}{An integer. Number of nearest neighbor that are used
to generate the new examples of the minority class.}

\item{skip}{A logical. Should the step be skipped when the
recipe is baked by \code{\link[recipes:bake]{bake()}}? While all operations are baked
when \code{\link[recipes:prep]{prep()}} is run, some operations may not be able to be
conducted on new data (e.g. processing the outcome variable(s)).
Care should be taken when using \code{skip = TRUE} as it may affect
the computations for subsequent operations.}

\item{seed}{An integer that will be used as the seed when
applied.}

\item{id}{A character string that is unique to this step to identify it.}
}
\value{
An updated version of \code{recipe} with the new step
added to the sequence of existing steps (if any). For the
\code{tidy} method, a tibble with columns \code{terms} which is
the variable used to sample.
}
\description{
\code{step_adasyn} creates a \emph{specification} of a recipe
step that generates synthetic positive instances using ADASYN algorithm.
}
\details{
All columns in the data are sampled and returned by \code{\link[=juice]{juice()}}
and \code{\link[=bake]{bake()}}.

All columns used in this step must be numeric with no missing data.

When used in modeling, users should strongly consider using the
option \code{skip = TRUE} so that the extra sampling is \emph{not}
conducted outside of the training set.
}
\section{Tidying}{
When you \code{\link[=tidy.recipe]{tidy()}} this step, a tibble with columns \code{terms}
(the selectors or variables selected) will be returned.
}

\section{Case weights}{


The underlying operation does not allow for case weights.
}

\examples{
library(recipes)
library(modeldata)
data(hpc_data)

hpc_data0 <- hpc_data \%>\%
  select(-protocol, -day)

orig <- count(hpc_data0, class, name = "orig")
orig

up_rec <- recipe(class ~ ., data = hpc_data0) \%>\%
  # Bring the minority levels up to about 1000 each
  # 1000/2211 is approx 0.4523
  step_adasyn(class, over_ratio = 0.4523) \%>\%
  prep()

training <- up_rec \%>\%
  bake(new_data = NULL) \%>\%
  count(class, name = "training")
training

# Since `skip` defaults to TRUE, baking the step has no effect
baked <- up_rec \%>\%
  bake(new_data = hpc_data0) \%>\%
  count(class, name = "baked")
baked

# Note that if the original data contained more rows than the
# target n (= ratio * majority_n), the data are left alone:
orig \%>\%
  left_join(training, by = "class") \%>\%
  left_join(baked, by = "class")

library(ggplot2)

ggplot(circle_example, aes(x, y, color = class)) +
  geom_point() +
  labs(title = "Without ADASYN")

recipe(class ~ x + y, data = circle_example) \%>\%
  step_adasyn(class) \%>\%
  prep() \%>\%
  bake(new_data = NULL) \%>\%
  ggplot(aes(x, y, color = class)) +
  geom_point() +
  labs(title = "With ADASYN")
}
\references{
He, H., Bai, Y., Garcia, E. and Li, S. 2008. ADASYN: Adaptive
synthetic sampling approach for imbalanced learning. Proceedings of
IJCNN 2008. (IEEE World Congress on Computational Intelligence). IEEE
International Joint Conference. pp.1322-1328.
}
\seealso{
\code{\link[=adasyn]{adasyn()}} for direct implementation

Other Steps for over-sampling: 
\code{\link{step_bsmote}()},
\code{\link{step_rose}()},
\code{\link{step_smotenc}()},
\code{\link{step_smote}()},
\code{\link{step_upsample}()}
}
\concept{Steps for over-sampling}
