% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tensorBF.R
\name{tensorBF}
\alias{tensorBF}
\title{Bayesian Factorization of a Tensor}
\usage{
tensorBF(Y, method = "CP", K = NULL, opts = NULL, fiberCentering = NULL,
  slabScaling = NULL, noiseProp = c(0.5, 0.5))
}
\arguments{
\item{Y}{is a three-mode tensor to be factorized.}

\item{method}{the factorization method. Currently only "CP" (default) is supported.}

\item{K}{The number of components (i.e. latent variables or factors). Recommended to be
  set somewhat higher than the expected component number, so that the method
  can determine the model complexity by prunning excessive components
  (default: 20\% of the sum of lower two dimensions).
  High values result in high CPU time.

  NOTE: Adjust parameter noiseProp if sufficiently
  large values of K do not lead to a model with pruned components.}

\item{opts}{List of model options; see function \code{\link{getDefaultOpts}} for details and default.}

\item{fiberCentering}{the mode for which fibers are to be centered at zero (default = NULL).
Fiber is analogous to a vector in a particular mode.
Fiber centering and Slab scaling are the recommended normalizations for a tensor.
For details see the provided normalization functions and the references therein.}

\item{slabScaling}{the mode for which slabs are to be scaled to unit variance (default = NULL).
Slab is analogous to a matrix in a particular mode.
Alternativly, you can preprocess the data using the provided normalization functions.}

\item{noiseProp}{c(prop,conf); sets an informative noise prior for tensorBF.
The model sets the noise prior such that the expected proportion of
variance explained by noise is defined by this parameter. It is recommended when
the standard prior from \code{\link{getDefaultOpts}} seems to overfit the model
by not prunning any component with high initial K. Use NULL to switch off
informative noise prior.

- prop defines the proportion of total variance to be explained by noise (between 0.1 and 0.9),

- conf defines the confidence in the prior (between 0.1 and 10).


We suggest a default value of c(0.5,0.5) for real data sets.}
}
\value{
A list containing model parameters.
  For key parameters, the final posterior sample ordered w.r.t. component variance is provided to aid in
  initial checks; all the posterior samples should be used for model
  analysis. The list elements are:
\item{K}{The number of learned components. If this value is not less then the input argument K, the model should be rerun with a larger K or use the noiseProp parameter.}
\item{X}{a matrix of \eqn{N \times K} dimensions, containing the last Gibbs sample of the first-mode latent variables.}
\item{W}{a matrix of \eqn{D \times K} dimensions, containing the last Gibbs sample of the second-mode latent variables.}
\item{U}{a matrix of \eqn{L \times K} dimensions, containing the last Gibbs sample of the third-mode latent variables.}
\item{tau}{The last sample of noise precision.}
and the following elements:
\item{posterior}{the posterior samples of model parameters (X,U,W,Z,tau).}
\item{cost}{The likelihood of all the posterior samples.}
\item{opts}{The options used to run the model.}
\item{conv}{An estimate of the convergence of the model, based on reconstruction
 of data using the Geweke diagnostic. Values significantly above 0.05 occur when
 model has not converged and should therefore be rerun with a higher value of iter.burnin in \code{\link{getDefaultOpts}}.}
\item{pre}{A list of centering and scaling values used to transform the data, if any. Else an empty list.}
}
\description{
\code{tensorBF} implements the Bayesian factorization of a tensor.
}
\details{
Bayesian Tensor Factorization performs tri-linear (CP) factorization of a tensor.
The method automatically identifies the number of components,
given K is initialized to a large enough value, see arguments.
Missing values are supported and should be set as NA's in the data.
They will not affect the model parameters, and can be predicted
with function \code{\link{predictTensorBF}}, based on the observed values.
}
\examples{
#Data generation
K <- 2
X <- matrix(rnorm(20*K),20,K)
W <- matrix(rnorm(30*K),30,K)
U <- matrix(rnorm(3*K),3,K)
Y = 0
for(k in 1:K) Y <- Y + outer(outer(X[,k],W[,k]),U[,k])
 Y <- Y + array(rnorm(20*30*3,0,0.25),dim=c(20,30,3))

#Run the method with default options
\dontrun{res2 <- tensorBF(Y=Y)}

#Run the method with K=3 and iterations=1000
opts <- getDefaultOpts(); opts$iter.burnin = 1000
res1 <- tensorBF(Y=Y,K=3,opts=opts)

#Vary the user defined expected proportion of noise variance
#explained. c(0.2,1) represents 0.2 as the noise proportion
#and confidence of 1
\dontrun{res3 <- tensorBF(Y=Y,noiseProp=c(0.2,1))}

}

