% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geometry.R
\name{geo}
\alias{geo}
\alias{geo.character}
\alias{geo.list}
\alias{geo.t2d_geo}
\alias{geo.t2d_tin}
\alias{print.t2d_geo}
\title{Mesh / geometry  object}
\usage{
geo(x, fname, ...)

\method{geo}{character}(x, fname = NULL, ...)

\method{geo}{list}(x, fname = NULL, ...)

\method{geo}{t2d_geo}(x, fname = NULL, data = NULL, ...)

\method{geo}{t2d_tin}(x, fname = NULL, ..., dem, title = "")

\method{print}{t2d_geo}(x, ...)
}
\arguments{
\item{x}{Either: A \code{character} string providing the name of an existing geometry file;
a \code{list} with all required elements (see 'Value');
an object of class \code{t2d_geo};
an object of class \code{t2d_tin} (requires input \code{dem}).}

\item{fname}{\code{character}, name for the associated geometry file (can also be used to replace an existing entry).}

\item{...}{Arguments passed to or from other methods: \code{\link{read_geo}}
if \code{x} is a file name or \code{\link{interpol}} if \code{x} is a \code{t2d_tin} object.}

\item{data}{\code{list} with all required elements (see \code{Value}) that can
be given to update an existing object \code{x}.}

\item{dem}{If \code{x} is of type \code{t2d_tin}: a single DEM or a named list
with element \code{elevation} and further private variables from which the values will
be interpolated to the TIN mesh points. Each element will be forwarded to
\code{\link{interpol}}, argument \code{src} (see descriptions therein for supported
types). Private variables must be a list with elements \code{values} (the object passed
to \code{interpol}), \code{pars_interp} (variable-specific parameter \code{list} passed to
\code{interpol} with priority over \code{...}), and \code{unit} (the unit of the
variable). Parameters for interpolation include, e.g., the number of neighbours
\code{n}, etc.}

\item{title}{\code{character}, mesh title (not necessary, max. 72 characters).}
}
\value{
An object of class \code{t2d_geo} consisting of an attribute \code{file}
pointing to a specific geometry file and a \code{list} with the following elements:
\describe{
  \item{header}{General mesh information including title, precision (of numbers in
    the slf file in bytes), the numbers of mesh elements and points, and the number
    of points per mesh (3 in case of triangles which is the only supported value for now).}
  \item{tin}{A mesh object of class \code{t2d_tin}, see \link{tin}.}
  \item{elevation}{Values of mesh point elevations.}
  \item{privars}{A named \code{list} of additional, in TELEMAC termed 'private'
    variables with elements \code{values} and \code{unit}.}
}
}
\description{
Initialise a mesh (geometry) object for use within TELEMAC.
}
\note{
Also note the associated \code{\link{plot.t2d_geo}} method.
}
\examples{
\dontrun{
library(raster)
library(sf)
library(tidyverse)

# get a tin
bnd <- st_read(system.file("dem/boundary_lagos.gpkg", package = "telemac"))
tin_obj <- tin(list(boundary = bnd), s = 90, a = 100^2, q = 30)

# load raster
dem_rast <- raster(system.file("dem/dem_merit_lagos.tif", package = "telemac"))

# create a geo object (interpolates raster to mesh points)
geo_obj <- geo(tin_obj, dem = dem_rast)
geo_obj
str(geo_obj)

# adjust file name
geo_obj <- geo(geo_obj, fname = "geo.slf")
geo_obj

# plot: mesh elevations interpolated to grid with resolution s
plot(geo_obj, s = 30)

# add additional private variable (in this case Curve Numbers)
dem_priv <- list(
  # mandatory 'elevation' as raster object
  elevation = dem_rast,
  # additional variable 'cn' as (in this case) data.frame
  cn = list(values = as.data.frame(dem_rast, xy = TRUE, na.rm = TRUE) \%>\%
              select(x, y) \%>\%
              mutate(z = case_when(
                y > 740000 ~ 95,
                x > 534000 ~ 90,
                (x <= 534000) & (y <= 740000) ~ 80
              )),
            unit = "-",
            # nearest-neighbour interpolation of CN values
            pars_interp = list(n = 1))
)

geo_priv <- geo(tin_obj, dem = dem_priv)

geo_priv
str(geo_priv)
plot(geo_priv, s = 30, v = "cn")
}
}
