% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/regressionSummaries.r
\name{XLregresSummary}
\alias{XLregresSummary}
\title{Regression Summary Tables exported to a spreadsheet}
\usage{
XLregresSummary(wb, sheet, betas, SE = NULL, varnames = NULL, colid = 1:2,
  transfun = identity, title = NULL, effname = "Difference",
  confac = qnorm(0.975), roundig = 2, pfun = function(x) 2 *
  pnorm(-abs(x)), pround = 3, row1 = 1, col1 = 1, purge = FALSE)
}
\arguments{
\item{wb}{a \code{\link[XLConnect]{workbook-class}} object}

\item{sheet}{numeric or character: a worksheet name (character) or position (numeric) within \code{wb}.}

\item{betas}{numeric: a vector of point estimates, or a matrix containing estimates and standard errors in columns}

\item{SE}{numeric: a vector of standard-error estimates for the effects. If \code{NULL} (default), user needs to specify them via the \code{betas} matrix.}

\item{varnames}{character: a vector of effect names (column 1 of output table). If \code{NULL} (default), user needs to specify them via the \code{betas} matrix.}

\item{colid}{integer: vector of indices for the columns containing the point estimates and SEs, respectively. Used only if \code{betas} is a matrix.}

\item{transfun}{transformation function for \code{betas,SE}, to produce columns 2-3 of the output. Defaults to \code{\link{identity}}. use {\code{\link{exp}}} for odds ratio or relative risk.}

\item{title}{character: an optional overall title to the table. Default (\code{NULL}) is no title.}

\item{effname}{character: a string explaining what the effect stands for, e.g. "difference" (the default), "Odds Ratio", etc.}

\item{confac}{numeric: the proportionality factor for calculating confidence-intervals. Default produces 95\% Normal intervals.}

\item{roundig}{numeric: how many digits (after the decimal point) to round the effect estimate to?}

\item{pfun}{function used to calculate the p-value, based on the signal-to-noise ratio \code{betas/SE}. Default assumes two-sided Normal p-values.}

\item{pround}{numeric: how many digits (after the decimal point) to round the p-value to? P-values rounded down to zero will show up as "<" the smallest nonzero value, e.g. with the default \code{pround=3} p-values smaller than 0.0005 will show up as "<0.001".}

\item{row1, col1}{numeric: the first row and column occupied by the table (title included if relevant).}

\item{purge}{logical: should \code{sheet} be created anew, by first removing the previous copy if it exists? (default \code{FALSE})}
}
\value{
The function returns invisibly, after writing the data into \code{sheet}.
}
\description{
Takes regression effect estimates and the corresponding standard errors, transforms to "human scale" if requested, calculates confidence-intervals and p-values, and exports a standard formatted summary table to a spreadsheet.
}
\details{
This function produces a standard scientific-article regression summary table, given the raw regression output. The resulting table has 4 columns: effect name, its (optionally transformed) magnitude, a probabilistically symmetric confidence interval (likewise transformed), and p-value. The formatted table is exported to \code{sheet}, and the file is immediately saved.

The input can be provided as separate vectors of point estimates (\code{betas}) and standard errors (\code{SE}), or as a single matrix for \code{betas}. In the latter case, as a default the effect names will be \code{rownames(betas)}, unless a vector with more descriptive names is provided via \code{varnames}.

See the \code{\link{XLtwoWay}} help page, for behavior regarding new-sheet creation, overwriting, etc.
}
\note{
The default CI's are 95\% and Normal. P-values are also derived from the Normal. If you run any regression whose intervals are calculated differently (e.g., linear regression with not-huge sample size), make sure to change both \code{confac} and \code{pfun} accordingly, as is shown in the example.
}
\examples{
book4<-XLwriteOpen("attenu.xls") 

quakenames=c("Magnitude (Richter), per unit","Distance (log km), per x10")

# Ground acceleration as a function of magnitude and distance, all log scale.
quakemod1=summary(lm(log10(accel)~mag+log10(dist),data=attenu))


## Model-scale summaries; we don't care for the intercept.

XLregresSummary(book4,"ModelScale",varnames=quakenames,
                betas=quakemod1$coef[-1,1],SE=quakemod1$coef[-1,2],
                ,title="Log-Ground Acceleration Effects",
                confac=qt(0.975,179),pfun=function(x) 2*pt(-abs(x),df=179))

## Same thing, but using matrix input; no need to provide SE and names. 
## It is arguably still nicer to provide your own names - but could be a reproducibility risk. 
## Also, increasing the p-value resolution by changing 'pround'.

XLregresSummary(book4,"ModelScale",betas=quakemod1$coef[-1,],
                pround=6,title="Log-Ground Acceleration Effects",
                confac=qt(0.975,179),pfun=function(x) 2*pt(-abs(x),df=179),row1=8)

## Effects are arguably more meaningful as percent change. 
## So... still same model, but different summaries. 
## Also, note the combination of matrix input with names over-written via 'varnames':

XLregresSummary(book4,"PercentChange",varnames=quakenames,
                betas=quakemod1$coef[-1,],
                roundig=1,pround=6,title="Relative Ground Acceleration Effects",
                transfun=function(x) 100*(10^x-1),
                effname="Percent Change",confac=qt(0.975,179),pfun=function(x) 2*pt(-abs(x),df=179))

cat("Look for",paste(getwd(),"attenu.xls",sep='/'),"to see the results!\\n")

### lm() does not take account of station or event level grouping.
### So we use a mixed model, losing 16 data points w/no station data:
### Run this on your own... and ask the authors of "lme4" about p-values at your own risk :)

# library(lme4)
# quakemod2=lmer(log10(accel)~mag+log10(dist)+(1|event)+(1|station),data=attenu)
# 
# XLregresSummary(book4,"MixedModel",varnames=quakenames,betas=fixef(quakemod2)[-1],
# SE=sqrt(diag(vcov(quakemod2)))[-1],
# roundig=1,pround=6,
# title="Relative Ground Acceleration Effects",
# transfun=function(x) 100*(10^x-1),effname="Percent Change",
# confac=qt(0.975,160),pfun=function(x) 2*pt(-abs(x),df=160))

}
\author{
Assaf P. Oron \code{<assaf.oron.at.seattlechildrens.org>}
}

