#' Extract matched groups from a string
#'
#' Dependency-free drop-in alternative for `stringr::str_match()`.
#'
#' @source Adapted from the [stringr](https://stringr.tidyverse.org/) package.
#'
#' @param string Input vector.
#'   Either a character vector, or something coercible to one.
#'
#' @param pattern Pattern to look for.
#'
#'   The default interpretation is a regular expression,
#'   as described in [base::regex].
#'   Control options with [regex()].
#'
#'   Match a fixed string (i.e. by comparing only bytes), using [fixed()].
#'   This is fast, but approximate.
#'
#' @return A character matrix.
#'   The first column is the complete match,
#'   followed by one column for each capture group.
#' @export
#' @staticexport
str_match <- function(string, pattern) {
	if (length(string) == 0 || length(pattern) == 0) return(matrix(character(0)))

	is_fixed <- inherits(pattern, "stringr_fixed")

	matches <- mapply(
		function(string, pattern) {
			if (is.na(string) || is.na(pattern)) return(NA_character_)

			regmatches(
				x = string,
				m = regexec(
					pattern = pattern, text = string, perl = !is_fixed, fixed = is_fixed
				)
			)
		},
		string, pattern, USE.NAMES = FALSE
	)

	length <- max(lengths(matches))
	matches <- lapply(matches, `[`, seq_len(length))

	result <- do.call(rbind, matches)
	result[result == ""] <- NA_character_
	result
}
