#' @title Standardized Moderation Effect and Its Bootstrap CI in 'lavaan'
#'
#' @description Compute the standardized moderation effect in a structural
#'              equation model fitted by [lavaan::lavaan()] or its wrappers and
#'              form the nonparametric bootstrap confidence interval.
#'
#' @details
#' [stdmod_lavaan()] accepts a [lavaan::lavaan-class] object, the
#' structural equation model output returned
#' by [lavaan::lavaan()] and its wrappers (e.g, [lavaan::sem()]) and computes
#' the standardized moderation effect using the formula in the appendix of
#' Cheung, Cheung, Lau, Hui, and Vong (2022).
#'
#' The standard deviations of the focal variable (the variable with its effect
#' on the outcome variable being moderated), moderator, and outcome
#' variable (dependent variable) are computed from the implied
#' covariance matrix returned by
#' [lavaan::lavInspect()]. Therefore, models fitted to data sets with missing
#' data (e.g., with `missing = "fiml"`) are also supported.
#'
#' If nonparametric bootstrap confidence interval is requested with `R`
#' bootstrap samples, the model will be fitted `R` times to these samples,
#' and the standardized
#' moderation effect will be computed in each sample. This ensures that all
#' components used in the computation, including the standard deviations, are
#' also computed from the bootstrapping samples.
#'
#' Note that the computation can be slow because [lavaan::lavaan()] or its
#' wrappers
#' will be called
#' `R` times.
#'
#' @return
#' A list of class `stdmod_lavaan` with these elements:
#'
#'  - `stdmod`: The standardized moderation effect.
#'
#'  - `ci`: The nonparametric bootstrap confidence interval. `NA` if
#'            confidence interval not requested.
#'
#'  - `boot_out`: The raw output from [boot::boot()]. `NA` if
#'            confidence interval not requested.
#'
#'  - `fit`: The original fit object.
#'
#' @param fit The SEM output by [lavaan::lavaan()] or its wrappers.
#' @param x The name of the focal variable in the model, the variable with
#'          its effect on the outcome variable being moderated.
#' @param y The name of the outcome variable (dependent variable) in the model.
#' @param w The name of the moderator in the model.
#' @param x_w The name of the product term (x * w) in the model. It can be
#'            the variable generated by the colon operator, e.g., `"x:w"`,
#'            which is only in the model and not in the original data set.
#' @param boot_ci Boolean. Whether nonparametric bootstrapping will be
#'                conducted. Default is `FALSE`.
#' @param R The number of nonparametric bootstrapping samples. Default is 100.
#'          Set this to at least 2000 in actual use.
#' @param conf The level of confidence. Default is .95, i.e., 95%.
#' @param ... Optional arguments to be passed to [boot::boot()]. Parallel
#'            processing can be used by adding the appropriate arguments in
#'            [boot::boot()].
#'
#'
#' @author Shu Fai Cheung <https://orcid.org/0000-0002-9871-9448>
#'
#'
#' @references
#' Cheung, S. F., Cheung, S.-H., Lau, E. Y. Y., Hui, C. H., & Vong, W. N.
#' (2022) Improving an old way to measure moderation effect in standardized
#' units. Advance online publication. *Health Psychology*.
#' \doi{10.1037/hea0001188}
#'
#' @examples
#'
#' #Load a test data of 500 cases
#'
#' dat <- test_mod1
#' library(lavaan)
#' mod <-
#' "
#' med ~ iv + mod + iv:mod + cov1
#' dv ~ med + cov2
#' "
#' fit <- sem(mod, dat)
#'
#' # Compute the standardized moderation effect
#' out_noboot <- stdmod_lavaan(fit = fit,
#'                             x = "iv",
#'                             y = "med",
#'                             w = "mod",
#'                             x_w = "iv:mod")
#' out_noboot
#'
#' # Compute the standardized moderation effect and
#' # its percentile confidence interval using
#' # nonparametric bootstrapping
#' set.seed(8479075)
#' system.time(out_boot <- stdmod_lavaan(fit = fit,
#'                                       x = "iv",
#'                                       y = "med",
#'                                       w = "mod",
#'                                       x_w = "iv:mod",
#'                                       boot_ci = TRUE,
#'                                       R = 50))
#' # In real analysis, R should be at least 2000.
#' out_boot
#'
#' @export

stdmod_lavaan <- function(fit,
                          x,
                          y,
                          w,
                          x_w,
                          boot_ci = FALSE,
                          R = 100,
                          conf = 0.95, ...) {
    if (!requireNamespace("lavaan", quietly = TRUE)) {
        stop(paste("lavaan needs to be installed to run this function."))
      }
    boot_i <- boot_i_gen(fit = fit, x = x,
                                    y = y,
                                    w = w,
                                    x_w = x_w)
    dat_org <- lavaan::lavInspect(fit, "data")
    if (!boot_ci) {
        stdmod <- stdmod_from_fit(fit = fit,
                                  x = x,
                                  y = y,
                                  w = w,
                                  x_w = x_w)
        boot_out <- NA
        stdmod_ci <- NA
      } else {
        boot_out <- boot::boot(dat_org, boot_i, R = R, ...)
        stdmod <- boot_out$t0
        if (any(is.na(boot_out$t))) {
            tmp <- paste0("Fit not safe in at least one bootstrap ",
                          "samples, either with an error or with ",
                          "a warning message. Please check the original ",
                          "fit object, or set 'warn' to FALSE ",
                          "in lavaan if the warning can be ignored.")
            warning(tmp)
            cat(paste0("\nNumber of valid bootstrap estimates: ",
                           sum(!is.na(boot_out$t)), "\n"))
          }
        stdmod_ci <- boot::boot.ci(boot_out,
                                   type = "perc",
                                   conf = conf)$percent[4:5]
        names(stdmod_ci) <- c("lower_bound", "upper_bound")
      }
    out <- list(stdmod = stdmod,
                ci     = stdmod_ci,
                boot_out = boot_out)
    out$call <- match.call()
    out$fit <- fit
    class(out) <- c("stdmod_lavaan", class(out))
    return(out)
  }

boot_i_gen <- function(fit, x, y, w, x_w) {
  force(fit)
  force(x)
  force(y)
  force(w)
  force(x_w)
  function(dat, i = NULL) {
      if (is.null(i)) {
          dat_i <- dat
        } else {
          dat_i <- dat[i, ]
        }
      fit_test <- tryCatch(fit_i <- lavaan::update(fit,
                                       data = dat_i,
                                       se = "none",
                                       h1 = FALSE,
                                       baseline = FALSE,
                                       test = "standard"
                                       ),
                            error = function(e) e,
                            warning = function(e) e)
      if (inherits(fit_test, "error")) {
          print(fit_test)
          return(NA)
        }
      if (inherits(fit_test, "warning")) {
          # options_old <- options("warn" = -1)
          warning(fit_test)
          # options("warn" = options_old$warn)
          return(NA)
        }
      if (!inherits(fit_i, "lavaan")) {
          warning("Something's wrong. Fit failed.")
          return(NA)
        } else {
          x_w_std_i <- stdmod_from_fit(fit = fit_i,
                                       x = x,
                                       y = y,
                                       w = w,
                                       x_w = x_w)
          return(x_w_std_i)
        }
    }
  }

stdmod_from_fit <- function(fit, x, y, w, x_w) {
    fit_cov_implied <- lavaan::lavInspect(fit, "implied")
    x_sd <- sqrt(diag(fit_cov_implied$cov)[x])
    w_sd <- sqrt(diag(fit_cov_implied$cov)[w])
    y_sd <- sqrt(diag(fit_cov_implied$cov)[y])
    x_w_i <- lavaan::coef(fit)[paste0(y, "~", x_w)]
    x_w_std_i <- x_w_i * x_sd * w_sd / y_sd
    return(x_w_std_i)
  }