% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glm_methods.R
\name{standardize_glm_dr}
\alias{standardize_glm_dr}
\title{Get regression standardized doubly-robust estimates from a glm}
\usage{
standardize_glm_dr(
  formula_outcome,
  formula_exposure,
  data,
  values,
  ci_level = 0.95,
  ci_type = "plain",
  contrasts = NULL,
  family_outcome = "gaussian",
  family_exposure = "binomial",
  reference = NULL,
  transforms = NULL
)
}
\arguments{
\item{formula_outcome}{The formula which is used to fit the glm model for the outcome.}

\item{formula_exposure}{The formula which is used to fit the glm model for the exposure.
If not \code{NULL},
a doubly robust estimator of the standardized estimator is used.}

\item{data}{The data.}

\item{values}{A named list or data.frame specifying the variables and values
at which marginal means of the outcome will be estimated.}

\item{ci_level}{Coverage probability of confidence intervals.}

\item{ci_type}{A string, indicating the type of confidence intervals.
Either "plain", which gives untransformed intervals, or "log", which gives
log-transformed intervals.}

\item{contrasts}{A vector of contrasts in the following format:
If set to \code{"difference"} or \code{"ratio"}, then \eqn{\psi(x)-\psi(x_0)}
or \eqn{\psi(x) / \psi(x_0)} are constructed, where \eqn{x_0} is a reference
level specified by the \code{reference} argument. Has to be \code{NULL}
if no references are specified.}

\item{family_outcome}{The family argument which is used to fit the glm model for the outcome.}

\item{family_exposure}{The family argument which is used to fit the glm model for the exposure.}

\item{reference}{A vector of reference levels in the following format:
If \code{contrasts} is not \code{NULL}, the desired reference level(s). This
must be a vector or list the same length as \code{contrasts}, and if not named,
it is assumed that the order is as specified in contrasts.}

\item{transforms}{A vector of transforms in the following format:
If set to \code{"log"}, \code{"logit"}, or \code{"odds"}, the standardized
mean \eqn{\theta(x)} is transformed into \eqn{\psi(x)=\log\{\theta(x)\}},
\eqn{\psi(x)=\log[\theta(x)/\{1-\theta(x)\}]}, or
\eqn{\psi(x)=\theta(x)/\{1-\theta(x)\}}, respectively.
If the vector is \code{NULL}, then \eqn{\psi(x)=\theta(x)}.}
}
\value{
An object of class \code{std_glm}.
This is basically a list with components estimates and covariance estimates in \code{res}.
Results for transformations, contrasts, references are stored in \code{res_contrasts}.
Obtain numeric results in a data frame with the \link{tidy} function.
}
\description{
Get regression standardized doubly-robust estimates from a glm
}
\details{
\code{standardize_glm_dr} performs regression standardization
in generalized linear models, see e.g., documentation for \code{standardize_glm_dr}. Specifically,
this version uses a doubly robust estimator for standardization, meaning inference is valid
when either the outcome regression or the exposure model is correctly specified
and there is no unmeasured confounding.
}
\examples{

# doubly robust estimator
# needs to correctly specify either the outcome model or the exposure model
# for confounding
# NOTE: only works with binary exposures
data <- AF::clslowbwt
x <- standardize_glm_dr(
  formula_outcome = bwt ~ smoker * (race + age + lwt) + I(age^2) + I(lwt^2),
  formula_exposure = smoker ~ race * age * lwt + I(age^2) + I(lwt^2),
  family_outcome = "gaussian",
  family_exposure = "binomial",
  data = data,
  values = list(smoker = c(0, 1)), contrasts = "difference", reference = 0
)

set.seed(6)
n <- 100
Z <- rnorm(n)
X <- rbinom(n, 1, prob = (1 + exp(Z))^(-1))
Y <- rbinom(n, 1, prob = (1 + exp(X + Z))^(-1))
dd <- data.frame(Z, X, Y)
x <- standardize_glm_dr(
  formula_outcome = Y ~ X * Z, formula_exposure = X ~ Z,
  family_outcome = "binomial",
  data = dd,
  values = list(X = 0:1), reference = 0,
  contrasts = c("difference"), transforms = c("odds")
)

}
\references{
Gabriel E.E., Sachs, M.C., Martinussen T., Waernbaum I.,
Goetghebeur E., Vansteelandt S., Sjölander A. (2024),
Inverse probability of treatment weighting with
generalized linear outcome models for doubly robust estimation.
\emph{Statistics in Medicine}, \bold{43}(3):534--547.
}
