% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitTD.R
\name{fitTD}
\alias{fitTD}
\title{Fit single trial mixed model}
\usage{
fitTD(
  TD,
  trials = names(TD),
  design = NULL,
  traits,
  what = c("fixed", "random"),
  covariates = NULL,
  useCheckId = FALSE,
  spatial = FALSE,
  engine = NA,
  control = NULL,
  progress = FALSE,
  ...
)
}
\arguments{
\item{TD}{An object of class \code{\link{TD}}.}

\item{trials}{A character vector specifying the trials for which the models
should be fitted.}

\item{design}{A character string specifying the experimental design. Either
"ibd" (incomplete block design), "res.ibd" (resolvable incomplete block
design), "rcbd" (randomized complete block design), "rowcol" (row column
design) or "res.rowcol" (resolvable row column design). Can be ignored when
the trial design is specified in the meta data (see \code{\link{setMeta}}).}

\item{traits}{A character vector specifying the traits for which the models
should be fitted.}

\item{what}{A character vector specifying whether "genotype" should
be fitted as fixed or random effect. If not specified, both models are
fitted.}

\item{covariates}{A character vector specifying covariates to be fitted as
extra fixed effects in the model.}

\item{useCheckId}{Should checkId be used as a fixed effect in the model?\cr
If \code{TRUE}, \code{TD} has to contain a column 'checkId'. Using checkId
as fixed effect can only be done when genotype is fitted as a random effect
in the model.}

\item{spatial}{Should spatial models be tried? Spatial models can
only be fitted with SpATS and asreml. If SpATS is used for modeling, only
spatial models can be fitted and spatial is always set to \code{TRUE}. If
asreml is used, fitting spatial models is optional.}

\item{engine}{A character string specifying the name of the mixed modeling
engine to use, either "SpATS", "lme4" or "asreml." For spatial models,
"SpaTS" is used as default, for other models "lme4".}

\item{control}{An optional list with control parameters to be passed to the
actual fitting functions. Currently \code{nSeg} and \code{nestDiv} are valid
parameters when fitting a model using SpATS. They pass a value to nseg and
nest.div in \code{\link[SpATS]{PSANOVA}} respectively. For \code{nSeg} also a
named list can be supplied containing values for nSeg per trial.\cr
\code{criterion} is a valid parameter when fitting a spatial model using
asreml. It may be used to pass a goodness-of-fit criterion for comparing
different spatial models. See also in details. Other parameters are ignored.}

\item{progress}{Should the progress of the modeling be printed. If
\code{TRUE}, for every trial a line is output indicating the traits fitted
for the particular trial.}

\item{...}{Further arguments to be passed to \code{SpATS}, \code{lme4} or
\code{asreml}.}
}
\value{
An object of class \code{STA}, a list containing, per trial
that has been analyzed, a list of:
\item{mRand}{A list of models with fitted with genotype as random effect.}
\item{mFix}{A list of models fitted with genotype as fixed effect.}
\item{TD}{An object of class \code{\link{TD}} containing the data on which
\code{mRand} and \code{mFix} are based.}
\item{traits}{A character vector indicating the traits for which the models
are fitted.}
\item{design}{A character string containing the design of the trial.
(see \code{\link{fitTD}} for the possible designs).}
\item{spatial}{A character string indicating the spatial part of the model.
\code{FALSE} if no spatial design has been used.}
\item{engine}{A character string containing the engine used for the
analysis.}
\item{predicted}{A character string indicating the variable that has been
predicted.}
\item{sumTab}{A data.frame with a summary table for the spatial models tried
when \code{engine = "asreml"} and \code{spatial = TRUE}}
}
\description{
Perform REML analysis given a specific experimental design using either
SpATS, lme4 or asreml. SpATS is used as a default method when row
coordinates (rowCoord) and column coordinates (colCoord) are present, lme4
otherwise.
See details for the exact models fitted.
}
\details{
The actual model fitted depends on the design. For the supported designs, the
following models are used:\tabular{lll}{
   Design \tab Code \tab Model fitted \cr
   incomplete block design \tab ibd \tab trait = \strong{subBlock} + genotype + \eqn{\epsilon} \cr
   resolvable incomplete block design \tab res.ibd \tab trait = \emph{repId} + \strong{repId:subBlock} + genotype + \eqn{\epsilon} \cr
   randomized complete block design \tab rcbd \tab trait = \emph{repId} + genotype + \eqn{\epsilon} \cr
   row column design \tab rowcol \tab trait = \strong{rowId} + \strong{colId} + genotype + \eqn{\epsilon} \cr
   resolvable row column design \tab res.rowcol \tab trait = \emph{repId} + \strong{repId:rowId} + \strong{repId:colId} + genotype + \eqn{\epsilon} \cr
}


In the models above, fixed effects are indicated in \emph{italics} whereas random
effects are indicated in \strong{bold}. genotype can be fitted as fixed or as
random effect depending on the value of the parameter \code{what}. Extra
fixed effects may be fitted using the parameter \code{covariates}.\cr\cr
If \code{SpATS} is used as modeling engine, an extra spatial term is always
included  in the model. This term is constructed using the function
\code{\link[SpATS]{PSANOVA}} from the SpATS package as\cr
\code{PSANOVA(colCoord, rowCoord, nseg = nSeg, nest.div = 2)}
where\cr \code{nSeg = (number of columns / 2, number of rows / 2)}. nseg and
nest.div can be modified using the \code{control} parameter.\cr\cr
When \code{asreml} is used for modeling and \code{spatial} is \code{TRUE}
seven models are fitted with different random terms and covariance structure.
The best model is determined based on a goodness-of-fit criterion, either
AIC or BIC. This can be set using the control parameter \code{criterion},
default is AIC.
The fitted random terms depend on the structure of the data. If the trial
has a regular structure, i.e. all replicates appear the same amount of times
in the trial, the following combinations of random and spatial terms are
fitted:\tabular{ll}{
   Random part \tab Spatial part \cr
   random effects based on design \tab none \cr
   random effects based on design \tab AR1(rowId):colId \cr
   random effects based on design \tab rowId:AR1(colId) \cr
   random effects based on design \tab AR1(rowId):ar1(colId) \cr
   random effects based on design + nugget \tab AR1(rowId):colId \cr
   random effects based on design + nugget \tab rowId:AR1(colId) \cr
   random effects based on design + nugget \tab AR1(rowId):AR1(colId) \cr
}


If the design is not regular the following combinations of random and spatial
terms are fitted:\tabular{ll}{
   Random part \tab Spatial part \cr
   random effects based on design \tab none \cr
   random effects based on design \tab exp(rowCoord):colCoord \cr
   random effects based on design \tab rowCoord:exp(colCoord) \cr
   random effects based on design \tab iexp(rowCoord, colCoord) \cr
   random effects based on design  + nugget \tab exp(rowCoord):colCoord \cr
   random effects based on design + nugget \tab rowCoord:exp(colCoord) \cr
   random effects based on design + nugget \tab iexp(rowCoord,colCoord) \cr
}
}
\examples{
\donttest{
## Fit model using lme4.
lmeMod <- fitTD(TD = TDHeat05,
                design = "ibd",
                traits = "yield",
                what = "fixed",
                engine = "lme4")

## Summarize results.
summary(lmeMod)

## Create base plots of the results.
plot(lmeMod)

## Create a pdf report summarizing results.
report(lmeMod,
       outfile = tempfile(fileext = ".pdf"),
       what = "fixed")

## Fit model using SpATS.
spaMod <- fitTD(TD = TDHeat05,
                design = "res.rowcol",
                traits = "yield",
                what = "fixed")
summary(spaMod)

## Create spatial plots of the results.
plot(spaMod, plotType = "spatial")

## Fit model using SpATS.
## Specify the number of segments to use in PSANOVA
spaMod2 <- fitTD(TD = TDHeat05,
                 design = "res.rowcol",
                 traits = "yield",
                 what = "fixed",
                 control = list(nSeg = c(13, 26)))

## Create a pdf report summarizing results.
report(spaMod, outfile = tempfile(fileext = ".pdf"), what = "fixed")

## Fit model using asreml.
if (requireNamespace("asreml", quietly = TRUE)) {
  asrMod <- fitTD(TD = TDHeat05,
                  design = "res.rowcol",
                  traits = "yield",
                  what = "fixed",
                  engine = "asreml")
  summary(asrMod)

  report(asrMod,
         outfile = tempfile(fileext = ".pdf"),
         what = "fixed")
}
}
}
\references{
Maria Xose Rodriguez-Alvarez, Martin P. Boer, Fred A. van Eeuwijk, Paul H.C.
Eilers (2017). Correcting for spatial heterogeneity in plant breeding
experiments with P-splines. Spatial Statistics
\doi{10.1016/j.spasta.2017.10.003}

Butler, D. G., et al. (2010). Analysis of Mixed Models for S language
environments: ASReml-R reference manual. Brisbane, DPI Publications

Douglas Bates, Martin Maechler, Ben Bolker, Steve Walker (2015). Fitting
Linear Mixed-Effects Models Using lme4. Journal of Statistical Software,
67(1), 1-48. \url{https://www.jstatsoft.org/article/view/v067i01/0}.
}
