\name{colorgram}
\alias{colorgram}
\title{ Draw a colorgram (heatmap) of a matrix }
\description{
  Plot a visual representation of a numeric matrix using colors to indicate values.
}
\usage{
colorgram(x = NULL, y = NULL, z = NULL, zsize = 1, 
          map, nz = 10, breaks = pretty, base = NA, colFn = jet,
          key = hkey, key.args = list(), 
          xlab = NULL, ylab = NULL, zlab = NULL, 
          outlier = NULL, ...) 
}
\arguments{
  \item{x, y}{ Locations of grid lines at which the values in z are measured. 
                These must be finite, non-missing and in (strictly) ascending order.  (see Details below)}
  \item{z}{ A numeric matrix containing the values to be visualized as colors (NAs are allowed). Note that x can be used instead of z for convenience. }
  \item{zsize}{ A numeric matrix specifying the relative size of each rectangle. }
  \item{map}{ A list, as generated by \code{\link{makecmap}}.  If missing, a color map is generated automatically. }
  \item{nz, breaks, base, colFn}{ Arguments passed to \code{\link{makecmap}}, if \code{map} is missing. }
  \item{key}{ A function to draw a color key, such as \code{\link{hkey}} or \code{\link{vkey}}. }
  \item{key.args}{ Arguments passed to the function given by \code{key}.  }
  \item{xlab, ylab}{ Labels for axes. }
  \item{zlab}{ Label for the color key. }
  \item{outlier}{ Color for values outside the \code{map} domain.  
                  If NULL, values falling outside the map domain will generate an error. }
  \item{\dots}{ Further arguments passed to \code{\link{cimage}}. }
}
\details{
  This function assigns colors to the elements of a matrix and plots it using \code{\link{cimage}}.  

  Data can be passed to this function in any format recognized by \code{\link{xyzmat.coords}}.
  
  \code{colorgram} is somewhat similar to \code{\link[graphics]{image}}.  However, \code{colorgram} adds the following functionality:  1. The value-to-color mapping can be specified (thus allowing unequal bin sizes).  2. A color key can be added, optionally.  3. A color can be specified for missing values.  4. The size of each grid rectangle can be adjusted to convey additional information.
  
  Two color key functions are provided in the beeswarm package: 1) \code{\link{hkey}} draws a horizontal key, in the lower-left corner by default. 2) \code{\link{vkey}}) draws a vertical key, in the lower-right corner by default. The latter usually looks better if the right-hand margin is increased. 
  
}
\value{
  Invisibly, \code{map}.
}
\seealso{  
If this is not quite what you are looking for, consider \code{\link[graphics]{image}}, \code{\link[graphics]{filled.contour}}, or \code{\link[lattice]{levelplot}}.  Also \code{\link[plotrix]{color2D.matplot}} in the \pkg{plotrix} package.
}
\examples{

  ## median Petal.Length as function of Sepal.Length and Sepal.Width
  pl <- matapply( iris[,1:3], FUN = median, nx = 20, ny = 15 )

  ## Draw a colorgram with the default horizontal color key  
  colorgram(pl, main = 'iris')

  ## ... or with the vertical color key  
  colorgram(pl, main = 'iris', key = vkey)

  ## ... add margin space to improve legibility 
  op <- par(mar = c(5,4,4,4)+0.1)
  colorgram(pl, main = 'iris', key = vkey, 
    key.args = list(skip = 2), zlab = 'Petal\nlength')
  par(op)
  
  ##  Here is the example from \code{\link{persp}}
  x <- seq(-10, 10, length= 30)
  y <- x
  f <- function(x,y) { r <- sqrt(x^2+y^2); 10 * sin(r)/(r) }
  z <- outer(x, y, f)
  colorgram(x, y, z)

  ## ... and with a slight fix to the key:
  colorgram(x, y, z, key.args = list(wh = c(1, 4, 14)))

  ## We could also make more space for the key:
  op <- par(mar = c(7,4,4,2)+0.1)
  colorgram(x, y, z, key.args = list(stretch = 3))
  par(op)
  
  ## Here are some alternatives to colorgram  
  persp(x, y, z, theta = 30, phi = 30, expand = 0.5, col = "lightblue")
  image(x, y, z)  
  contour(x, y, z)

  ## Use 'xlabels' and 'ylabels' to create categorical axes
  colorgram(t(mtcars[,c(2,8:11)]), colFn = heat, 
    xlabels = TRUE, ylabels = TRUE, 
    xlab = NA, ylab = NA, zlab = 'Value', 
    main = 'Motor car specifications', las = 1)
  
}
\keyword{ hplot }
