% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sparsegl.R
\name{sparsegl}
\alias{sparsegl}
\title{Regularization paths for sparse group-lasso models}
\usage{
sparsegl(
  x,
  y,
  group = NULL,
  family = c("gaussian", "binomial"),
  nlambda = 100,
  lambda.factor = ifelse(nobs < nvars, 0.01, 1e-04),
  lambda = NULL,
  pf_group = sqrt(bs),
  pf_sparse = rep(1, nvars),
  intercept = TRUE,
  asparse = 0.05,
  standardize = TRUE,
  lower_bnd = -Inf,
  upper_bnd = Inf,
  weights = NULL,
  offset = NULL,
  warm = NULL,
  trace_it = 0,
  dfmax = as.integer(max(group)) + 1L,
  pmax = min(dfmax * 1.2, as.integer(max(group))),
  eps = 1e-08,
  maxit = 3e+06
)
}
\arguments{
\item{x}{Double. A matrix of predictors, of dimension
\eqn{n \times p}{n * p}; each row
is a vector of measurements and each column is a feature. Objects of class
\code{\link[Matrix:sparseMatrix]{Matrix::sparseMatrix}} are supported.}

\item{y}{Double/Integer/Factor. The response variable.
Quantitative for \code{family="gaussian"} and for other exponential families.
If \code{family="binomial"} should be either a factor with two levels or
a vector of integers taking 2 unique values. For a factor, the last level
in alphabetical order is the target class.}

\item{group}{Integer. A vector of consecutive integers describing the
grouping of the coefficients (see example below).}

\item{family}{Character or function. Specifies the generalized linear model
to use. Valid options are:
\itemize{
\item \code{"gaussian"} - least squares loss (regression, the default),
\item \code{"binomial"} - logistic loss (classification)
}

For any other type, a valid \code{\link[stats:family]{stats::family()}} object may be passed. Note
that these will generally be much slower to estimate than the built-in
options passed as strings. So for example, \code{family = "gaussian"} and
\code{family = gaussian()} will produce the same results, but the first
will be much faster.}

\item{nlambda}{The number of \code{lambda} values - default is 100.}

\item{lambda.factor}{A multiplicative factor for the minimal lambda in the
\code{lambda} sequence, where \code{min(lambda) = lambda.factor * max(lambda)}.
\code{max(lambda)} is the smallest value of \code{lambda} for which all coefficients
are zero. The default depends on the relationship between \eqn{n}
(the number of rows in the matrix of predictors) and \eqn{p}
(the number of predictors). If \eqn{n \geq p}, the
default is \code{0.0001}.  If \eqn{n < p}, the default is \code{0.01}.
A very small value of \code{lambda.factor} will lead to a
saturated fit. This argument has no effect if there is user-defined
\code{lambda} sequence.}

\item{lambda}{A user supplied \code{lambda} sequence. The default, \code{NULL}
results in an automatic computation based on \code{nlambda}, the smallest value
of \code{lambda} that would give the null model (all coefficient estimates equal
to zero), and \code{lambda.factor}. Supplying a value of \code{lambda} overrides
this behaviour. It is likely better to supply a
decreasing sequence of \code{lambda} values than a single (small) value. If
supplied, the user-defined \code{lambda} sequence is automatically sorted in
decreasing order.}

\item{pf_group}{Penalty factor on the groups, a vector of the same
length as the total number of groups. Separate penalty weights can be applied
to each group of \eqn{\beta}{beta's}s to allow differential shrinkage.
Can be 0 for some
groups, which implies no shrinkage, and results in that group always being
included in the model (depending on \code{pf_sparse}). Default value for each
entry is the square-root of the corresponding size of each group.}

\item{pf_sparse}{Penalty factor on l1-norm, a vector the same length as the
total number of columns in \code{x}. Each value corresponds to one predictor
Can be 0 for some predictors, which
implies that predictor will be receive only the group penalty.}

\item{intercept}{Whether to include intercept in the model. Default is TRUE.}

\item{asparse}{The relative weight to put on the \eqn{\ell_1}-norm in
sparse group lasso. Default is \code{0.05} (resulting in \code{0.95} on the
\eqn{\ell_2}-norm).}

\item{standardize}{Logical flag for variable standardization (scaling) prior
to fitting the model. Default is TRUE.}

\item{lower_bnd}{Lower bound for coefficient values, a vector in length of 1
or of length the number of groups. Must be non-positive numbers only.
Default value for each entry is \code{-Inf}.}

\item{upper_bnd}{Upper for coefficient values, a vector in length of 1
or of length the number of groups. Must be non-negative numbers only.
Default value for each entry is \code{Inf}.}

\item{weights}{Double vector. Optional observation weights. These can
only be used with a \code{\link[stats:family]{stats::family()}} object.}

\item{offset}{Double vector. Optional offset (constant predictor without a
corresponding coefficient). These can only be used with a
\code{\link[stats:family]{stats::family()}} object.}

\item{warm}{List created with \code{\link[=make_irls_warmup]{make_irls_warmup()}}. These can only be used
with a \code{\link[stats:family]{stats::family()}} object, and is not typically necessary even then.}

\item{trace_it}{Scalar integer. Larger values print more output during
the irls loop. Typical values are \code{0} (no printing), \code{1} (some printing
and a progress bar), and \code{2} (more detailed printing).
These can only be used with a \code{\link[stats:family]{stats::family()}} object.}

\item{dfmax}{Limit the maximum number of groups in the model. Default is
no limit.}

\item{pmax}{Limit the maximum number of groups ever to be nonzero. For
example once a group enters the model, no matter how many times it exits or
re-enters model through the path, it will be counted only once.}

\item{eps}{Convergence termination tolerance. Defaults value is \code{1e-8}.}

\item{maxit}{Maximum number of outer-loop iterations allowed at fixed lambda
value. Default is \code{3e8}. If models do not converge, consider increasing
\code{maxit}.}
}
\value{
An object with S3 class \code{"sparsegl"}. Among the list components:
\itemize{
\item \code{call} The call that produced this object.
\item \code{b0} Intercept sequence of length \code{length(lambda)}.
\item \code{beta} A \code{p} x \code{length(lambda)} sparse matrix of coefficients.
\item \code{df} The number of features with nonzero coefficients for each value of
\code{lambda}.
\item \code{dim} Dimension of coefficient matrix.
\item \code{lambda} The actual sequence of \code{lambda} values used.
\item \code{npasses} Total number of iterations summed over all \code{lambda} values.
\item \code{jerr} Error flag, for warnings and errors, 0 if no error.
\item \code{group} A vector of consecutive integers describing the grouping of the
coefficients.
\item \code{nobs} The number of observations used to estimate the model.
}

If \code{sparsegl()} was called with a \code{\link[stats:family]{stats::family()}} method, this may also
contain information about the deviance and the family used in fitting.
}
\description{
Fits regularization paths for sparse group-lasso penalized learning problems at a
sequence of regularization parameters \code{lambda}.
Note that the objective function for least squares is
\deqn{RSS/(2n) + \lambda penalty}
Users can also tweak the penalty by choosing a different penalty factor.
}
\examples{
n <- 100
p <- 20
X <- matrix(rnorm(n * p), nrow = n)
eps <- rnorm(n)
beta_star <- c(rep(5, 5), c(5, -5, 2, 0, 0), rep(-5, 5), rep(0, (p - 15)))
y <- X \%*\% beta_star + eps
groups <- rep(1:(p / 5), each = 5)
fit <- sparsegl(X, y, group = groups)

yp <- rpois(n, abs(X \%*\% beta_star))
fit_pois <- sparsegl(X, yp, group = groups, family = poisson())
}
\seealso{
\code{\link[=cv.sparsegl]{cv.sparsegl()}} and the \code{\link[=plot.sparsegl]{plot()}},
\code{\link[=predict.sparsegl]{predict()}}, and \code{\link[=coef.sparsegl]{coef()}}
methods for \code{"sparsegl"} objects.
}
