% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stvcGLMexact.R
\name{stvcGLMexact}
\alias{stvcGLMexact}
\title{Bayesian spatially-temporally varying generalized linear model}
\usage{
stvcGLMexact(
  formula,
  data = parent.frame(),
  family,
  sp_coords,
  time_coords,
  cor.fn,
  process.type,
  sptParams,
  priors,
  boundary = 0.5,
  n.samples,
  loopd = FALSE,
  loopd.method = "exact",
  CV.K = 10,
  loopd.nMC = 500,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{formula}{a symbolic description of the regression model to be fit.
Variables in parenthesis are assigned spatially-temporally varying
coefficients. See examples.}

\item{data}{an optional data frame containing the variables in the model.
If not found in \code{data}, the variables are taken from
\code{environment(formula)}, typically the environment from which
\code{stvcGLMexact} is called.}

\item{family}{Specifies the distribution of the response as a member of the
exponential family. Supported options are \code{'poisson'}, \code{'binomial'} and
\code{'binary'}.}

\item{sp_coords}{an \eqn{n \times 2}{n x 2} matrix of the observation
spatial coordinates in \eqn{\mathbb{R}^2} (e.g., easting and northing).}

\item{time_coords}{an \eqn{n \times 1}{n x 1} matrix of the observation
temporal coordinates in \eqn{\mathcal{T} \subseteq [0, \infty)}.}

\item{cor.fn}{a quoted keyword that specifies the correlation function used
to model the spatial-temporal dependence structure among the observations.
Supported covariance model key words are: \code{'gneiting-decay'} (Gneiting
and Guttorp 2010). See below for details.}

\item{process.type}{a quoted keyword specifying the model for the
spatial-temporal process. Supported keywords are \code{'independent'} which
indicates independent processes for each varying coefficients characterized
by different process parameters, \code{independent.shared} implies independent
processes for the varying coefficients that shares common process parameters,
and \code{multivariate} implies correlated processes for the varying coefficients
modeled by a multivariate Gaussian process with an inverse-Wishart prior on
the correlation matrix. The input for \code{sptParams} and \code{priors} must be given
accordingly.}

\item{sptParams}{fixed values of spatial-temporal process parameters in
usually a list of length 2. If \code{cor.fn='gneiting-decay'}, then it is a list
of length 2 with tags \code{phi_s} and \code{phi_t}. If \code{process.type='independent'},
then \code{phi_s} and \code{phi_t} contain fixed values of the \eqn{r} spatial-temporal
processes, otherwise they will contain scalars. See examples below.}

\item{priors}{(optional) a list with each tag corresponding to a
hyperparameter name and containing hyperprior details. Valid tags include
\code{V.beta}, \code{nu.beta}, \code{nu.z}, \code{sigmaSq.xi} and \code{IW.scale}. Values of \code{nu.beta}
and \code{nu.z} must be at least 2.1. If not supplied, uses defaults.}

\item{boundary}{Specifies the boundary adjustment parameter. Must be a real
number between 0 and 1. Default is 0.5.}

\item{n.samples}{number of posterior samples to be generated.}

\item{loopd}{logical. If \code{loopd=TRUE}, returns leave-one-out predictive
densities, using method as given by \code{loopd.method}. Default is
\code{FALSE}.}

\item{loopd.method}{character. Ignored if \code{loopd=FALSE}. If \code{loopd=TRUE},
valid inputs are \code{'exact'}, \code{'CV'}. The option \code{'exact'} corresponds to exact
leave-one-out predictive densities which requires computation almost
equivalent to fitting the model \eqn{n} times. The options \code{'CV'} is faster
as it implements \eqn{K}-fold cross validation to find approximate
leave-one-out predictive densities (Vehtari \emph{et al.} 2017).}

\item{CV.K}{An integer between 10 and 20. Considered only if
\code{loopd.method='CV'}. Default is 10 (as recommended in Vehtari \emph{et. al} 2017).}

\item{loopd.nMC}{Number of Monte Carlo samples to be used to evaluate
leave-one-out predictive densities when \code{loopd.method} is set to either
'exact' or 'CV'.}

\item{verbose}{logical. If \code{verbose = TRUE}, prints model description.}

\item{...}{currently no additional argument.}
}
\value{
An object of class \code{stvcGLMexact}, which is a list with the
following tags -
\describe{
\item{priors}{details of the priors used, containing the values of the
boundary adjustment parameter (\code{boundary}), the variance parameter of the
fine-scale variation term (\code{simasq.xi}) and others.}
\item{samples}{a list of length 3, containing posterior samples of fixed
effects (\code{beta}), spatial-temporal effects (\code{z}) and the
fine-scale variation term (\code{xi}). The element with tag \code{z} will
again be a list of length \eqn{r}, each containing posterior samples of the
spatial-temporal random effects corresponding to each varying coefficient.}
\item{loopd}{If \code{loopd=TRUE}, contains leave-one-out predictive
densities.}
\item{model.params}{Values of the fixed parameters that includes \code{phi_s}
(spatial decay), \code{phi_t} (temporal smoothness).}
}
The return object might include additional data that can be used for
subsequent prediction and/or model fit evaluation.
}
\description{
Fits a Bayesian generalized linear model with
spatially-temporally varying coefficients under fixed values of
spatial-temporal process parameters and some auxiliary model parameters. The
output contains posterior samples of the fixed effects, spatial-temporal
random effects and, if required, finds leave-one-out predictive densities.
}
\details{
With this function, we fit a Bayesian hierarchical
spatially-temporally varying generalized linear model by sampling exactly
from the joint posterior distribution utilizing the generalized conjugate
multivariate distribution theory (Bradley and Clinch 2024). Suppose
\eqn{\chi = (\ell_1, \ldots, \ell_n)} denotes the \eqn{n} spatial-temporal
co-ordinates in \eqn{\mathcal{L} = \mathcal{S} \times \mathcal{T}}, the
response \eqn{y} is observed. Let \eqn{y(\ell)} be the outcome at the
co-ordinate \eqn{\ell} endowed with a probability law from the natural
exponential family, which we denote by
\deqn{
y(\ell) \sim \mathrm{EF}(x(\ell)^\top \beta + \tilde{x}(\ell)^\top z(\ell);
b(\ell), \psi)
}
for some positive parameter \eqn{b(\ell) > 0} and unit log partition function
\eqn{\psi}. Here, \eqn{\tilde{x}(\ell)} denotes covariates with
spatially-temporally varying coefficients We consider the following response
models based on the input supplied to the argument \code{family}.
\describe{
\item{\code{'poisson'}}{It considers point-referenced Poisson responses
\eqn{y(\ell) \sim \mathrm{Poisson}(e^{x(\ell)^\top \beta +
\tilde{x}(\ell)^\top z(\ell)})}. Here, \eqn{b(\ell) = 1} and
\eqn{\psi(t) = e^t}.}
\item{\code{'binomial'}}{It considers point-referenced binomial counts
\eqn{y(\ell) \sim \mathrm{Binomial}(m(\ell), \pi(\ell))} where, \eqn{m(\ell)}
denotes the total number of trials and probability of success
\eqn{\pi(\ell) = \mathrm{ilogit}(x(\ell)^\top \beta + \tilde{x}(\ell)^\top
z(\ell))} at spatial-temporal co-ordinate \eqn{\ell}. Here, \eqn{b = m(\ell)}
and \eqn{\psi(t) = \log(1+e^t)}.}
\item{\code{'binary'}}{It considers point-referenced binary data (0 or, 1) i.e.,
\eqn{y(\ell) \sim \mathrm{Bernoulli}(\pi(\ell))}, where probability of
success \eqn{\pi(\ell) = \mathrm{ilogit}(x(\ell)^\top \beta +
\tilde{x}(\ell)^\top z(\ell))} at spatial-temporal co-ordinate \eqn{\ell}.
Here, \eqn{b(\ell) = 1} and \eqn{\psi(t) = \log(1 + e^t)}.}
}
The hierarchical model is given as
\deqn{
\begin{aligned}
y(\ell_i) &\mid \beta, z, \xi \sim EF(x(\ell_i)^\top \beta +
\tilde{x}(\ell_i)^\top z(s_i) + \xi_i - \mu_i; b_i, \psi_y),
i = 1, \ldots, n\\
\xi &\mid \beta, z, \sigma^2_\xi, \alpha_\epsilon \sim
\mathrm{GCM_c}(\cdots),\\
\beta &\mid \sigma^2_\beta \sim N(0, \sigma^2_\beta V_\beta), \quad
\sigma^2_\beta \sim \mathrm{IG}(\nu_\beta/2, \nu_\beta/2)\\
z_j &\mid \sigma^2_{z_j} \sim N(0, \sigma^2_{z_j} R(\chi; \phi_s, \phi_t)),
\quad \sigma^2_{z_j} \sim \mathrm{IG}(\nu_z/2, \nu_z/2), j = 1, \ldots, r
\end{aligned}
}
where \eqn{\mu = (\mu_1, \ldots, \mu_n)^\top} denotes the discrepancy
parameter. We fix the spatial-temporal process parameters \eqn{\phi_s} and
\eqn{\phi_t} and the hyperparameters \eqn{V_\beta}, \eqn{\nu_\beta},
\eqn{\nu_z} and \eqn{\sigma^2_\xi}. The term \eqn{\xi} is known as the
fine-scale variation term which is given a conditional generalized conjugate
multivariate distribution as prior. For more details, see Pan \emph{et al.} 2024.
Default values for \eqn{V_\beta}, \eqn{\nu_\beta}, \eqn{\nu_z},
\eqn{\sigma^2_\xi} are diagonal with each diagonal element 100, 2.1, 2.1 and
0.1 respectively.
}
\examples{
data("sim_stvcPoisson")
dat <- sim_stvcPoisson[1:100, ]

# Fit a spatial-temporal varying coefficient Poisson GLM
mod1 <- stvcGLMexact(y ~ x1 + (x1), data = dat, family = "poisson",
                     sp_coords = as.matrix(dat[, c("s1", "s2")]),
                     time_coords = as.matrix(dat[, "t_coords"]),
                     cor.fn = "gneiting-decay",
                     process.type = "multivariate",
                     sptParams = list(phi_s = 1, phi_t = 1),
                     verbose = FALSE, n.samples = 100)
}
\references{
Bradley JR, Clinch M (2024). "Generating Independent Replicates
Directly from the Posterior Distribution for a Class of Spatial Hierarchical
Models." \emph{Journal of Computational and Graphical Statistics}, \strong{0}(0), 1-17.
\doi{10.1080/10618600.2024.2365728}.

T. Gneiting and P. Guttorp (2010). "Continuous-parameter
spatio-temporal processes." In \emph{A.E. Gelfand, P.J. Diggle, M. Fuentes, and
P Guttorp, editors, Handbook of Spatial Statistics}, Chapman & Hall CRC
Handbooks of Modern Statistical Methods, p 427–436. Taylor and Francis.

Pan S, Zhang L, Bradley JR, Banerjee S (2024). "Bayesian
Inference for Spatial-temporal Non-Gaussian Data Using Predictive Stacking."
\doi{10.48550/arXiv.2406.04655}.

Vehtari A, Gelman A, Gabry J (2017). "Practical Bayesian Model
Evaluation Using Leave-One-out Cross-Validation and WAIC."
\emph{Statistics and Computing}, \strong{27}(5), 1413-1432. ISSN 0960-3174.
\doi{10.1007/s11222-016-9696-4}.
}
\seealso{
\code{\link[=spGLMexact]{spGLMexact()}}
}
\author{
Soumyakanti Pan \href{mailto:span18@ucla.edu}{span18@ucla.edu}
}
