% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aovGeo.R
\name{aovGeo}
\alias{aovGeo}
\alias{aovGeo.spVariofitRCBD}
\title{Analysis of variance using a geostatistical model to handle spatial dependence}
\usage{
aovGeo(model, cutoff = 0.5, tol = 1e-3)

\method{aovGeo}{spVariofitRCBD}(model, cutoff = 0.5, tol = 0.001)
}
\arguments{
\item{model}{an object of class spVariofit.}

\item{cutoff}{a value in (0,1] interval representing the percentage of the maximum
distance adopted to estimate the variogram in the algorithm suggested by
Pontes & Oliveira (2004). See 'Details'.}

\item{tol}{the desired accuracy.}
}
\value{
\code{aovGeo} returns an object of \code{\link[base]{class}} "GEOanova".
The functions \code{\link[base]{summary}} and anova are used to obtain and print a summary
and analysis of variance table of the results.
An object of class "GEOanova" is a list containing the following components:

\item{DF}{degrees of freedom of coordinates (in presence of spatial trend), treatments, block (if RCBD), residual and total.}
\item{SS}{sum of squares of coordinates (in presence of spatial trend), treatments, block (if RCBD), residual and total.}
\item{MS}{mean of squares of coordinates (in presence of spatial trend), treatments, block (if RCBD), residual and total.}
\item{Fc}{F statistic calculated for coordinates (in presence of spatial trend), treatment and block (if RCBD).}
\item{p.value}{p-value associated to F statistic for coordinates (in presence of spatial trend), treatment and block (if RCBD).}
\item{residuals}{residuals extracted from the model}
\item{params}{variogram parameter estimates from Pontes & Oliveira (2004) algorithm.}
\item{type}{type of trend in the data. It can be "trend" or "cte".}
\item{model}{geostatistical model used to describe the spatial dependence.}
\item{data}{data set used in the analysis.}
\item{des.mat}{design matrix.}
\item{beta}{parameter estimates of the linear model taking into account the spatial dependence.}
\item{n}{number of observations.}
\item{vcov}{covariance matrix built taking into account the spatial dependence.}
\item{design}{character string indicating the name of the experimental design.}
}
\description{
Fit an analysis of variance model using geostatistics for modeling the
spatial dependence among the observations.
}
\details{
Three assumptions are made about the error in the analysis of variance (ANOVA):

1. The errors come from a normal distribution.

2. The errors have the same variance.

3. The errors are uncorrelated.

However, in many experiments, especially field trials, there is a type of correlation
generated by the sample locations known as spatial autocorrelation, and this condition
violates the independence assumption.

In that way, we need to regard this spatial autocorrelation and include it in the final
model. It could be done adopting a geostatistical model to characterize the spatial
variability among the observations directly in the covariance matrix.

The geostatistical modeling is based on the residuals of the standard model
(where the errors are assumed to be independent, uncorrelated and having a normal
distribution with mean 0 and constant variance). The basic idea is using them to
estimate the residuals of the spatially autocorrelated model in order to fit a
theoretical geostatistic model to build the covariance matrix.
As Pointed by Pontes & Oliveira (2004), this task can be done using the
following algorithm

1 - Extract the residuals from the standard model

2 - Fit a variogram based on residuals obtained in step 1.

3 - Fit a theoretical model to describe the spatial dependence observed in the variogram.

4 - On basis in the theoretical model fitted in step 3 and its parameter estimates,
create the covariance matrix.

5 - Estimate the residuals using the covariance matrix obtained in step 4 and use
them to create a variogram.

6 -  Fit a theoretical model to the residual variogram obtained in step 5 and use
its parameters estimates to build a new covariance matrix.

7 - Repeat 5 to 6 until convergence.

Step 1 is implemented in spVariog. Steps 2 and 3 are implemented in spVariofit.
aovGeo implements steps 4 to 7 and needs a cutoff argument to define the maximum
distance in the computation of the residual variogram described in step 6

In presence of spatial trend, the model is modified
in order to include the effect of the spatial coordinates.
}
\examples{
data("crd_simulated")

#Geodata object
geodados <- as.geodata(crd_simulated, coords.col = 1:2, data.col = 3,
                            covar.col = 4)
h_max <- summary(geodados)[[3]][[2]]
dist <- 0.6*h_max

# Computing the variogram
variograma <- spVariog(geodata = geodados,
                      trend = "cte", max.dist = dist, design = "crd",
                      scale = FALSE)

plot(variograma, ylab = "Semivariance", xlab = "Distance")

# Gaussian Model
ols <- spVariofit(variograma, cov.model = "gaussian", weights = "equal",
                  max.dist = dist)

lines(ols, col = 1)

# Compute the model and get the analysis of variance table
mod <- aovGeo(ols, cutoff = 0.6)
anova(mod)

}
\references{
NOGUEIRA, C. H., de LIMA, R. R., & de OLIVEIRA, M. S. (2013).
Aprimoramento da Análise de Variância: A Influência da Proximidade Espacial.
Rev. Bras. Biom, 31(3), 408-422.

NOGUEIRA, C.H., et al. (2015). Modelagem espacial na análise de um plantio
experimental de candeia. Rev. Bras. Biom., São Paulo, v.33, n.1, p.14-29.

Pontes, J. M., & de Oliveira, M. S. (2004).
An alternative proposal to the analysis of  field experiments using geostatistics.
Ciência e Agrotecnologia, 28(1), 135-141.

Gotway, C. A., & Cressie, N. A. (1990). A spatial analysis of variance applied
to soil water infiltration. Water Resources Research, 26(11), 2695-703.
}
