% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eval_design_survival_mc.R
\name{eval_design_survival_mc}
\alias{eval_design_survival_mc}
\title{Evaluate Power for Survival Design}
\usage{
eval_design_survival_mc(RunMatrix, model, alpha, nsim = 1000,
  distribution = "gaussian", censorpoint = NA, censortype = "right",
  rfunctionsurv = NULL, anticoef = NULL, delta = 2,
  contrasts = contr.sum, parallel = FALSE, detailedoutput = FALSE,
  progressBarUpdater = NULL, ...)
}
\arguments{
\item{RunMatrix}{The run matrix of the design. Internally, all numeric columns will be rescaled to [-1, +1].}

\item{model}{The statistical model used to fit the data.}

\item{alpha}{The type-I error.}

\item{nsim}{The number of simulations. Default 1000.}

\item{distribution}{Distribution of survival function to use when fitting the data. Valid choices are described
in the documentation for \code{survreg}. \emph{Supported} options are
"exponential", "lognormal", or "gaussian". Default "gaussian".}

\item{censorpoint}{The point after/before (for right-censored or left-censored data, respectively)
which data should be labelled as censored. Default NA for no censoring.}

\item{censortype}{The type of censoring (either "left" or "right"). Default "right".}

\item{rfunctionsurv}{Random number generator function. Should be a function of the form f(X,b), where X is the
model matrix and b are the anticipated coefficients. This function should return a \code{Surv} object from
the \code{survival} package. You do not need to provide this argument if \code{distribution} is one of
the supported choices and you are satisfied with the default behavior described below.}

\item{anticoef}{The anticipated coefficients for calculating the power. If missing, coefficients
will be automatically generated based on the \code{delta} argument.}

\item{delta}{The signal-to-noise ratio. For a gaussian model, this specifies the difference in
response between the highest and lowest levels of a factor (which are +1 and -1, respectively, after normalization).
If you do not specify \code{anticoef}, the anticipated coefficients will be half of \code{delta}. Default 2.}

\item{contrasts}{Function used to encode categorical variables in the model matrix. Default \code{contr.sum}.}

\item{parallel}{If TRUE, uses all cores available to speed up computation of power. Default FALSE.}

\item{detailedoutput}{If TRUE, return additional information about evaluation in results. Default FALSE.}

\item{progressBarUpdater}{Default NULL. Function called in non-parallel simulations that can be used to update external progress bar.}

\item{...}{Any additional arguments to be passed into the \code{survreg} function during fitting.}
}
\value{
A data frame consisting of the parameters and their powers. The parameter estimates from the simulations are
stored in the 'estimates' attribute. The 'modelmatrix' attribute contains the model matrix and the encoding used for
categorical factors. If you manually specify anticipated coefficients, do so in the order of the model matrix.
}
\description{
Evaluates power for an experimental design in which the response variable may be
right- or left-censored. Power is evaluated with a Monte Carlo simulation,
using the \code{survival} package and \code{survreg} to fit the data. Split-plot designs are not supported.
}
\details{
If not supplied by the user, \code{rfunctionsurv} will be generated based on the \code{distribution}
argument as follows:
\tabular{lr}{
\bold{distribution}  \tab \bold{generating function} \cr
"gaussian"                  \tab \code{rnorm(mean = X \%*\% b, sd=1)}           \cr
"exponential"               \tab \code{rexp(rate = exp(-X \%*\% b))}           \cr
"lognormal"                 \tab \code{rlnorm(meanlog= X \%*\% b, sdlog=1)}           \cr
}

In each case, if a simulated data point is past the censorpoint (greater than for right-censored, less than for
left-censored) it is marked as censored. See the examples below for how to construct your own function.
}
\examples{
#These examples focus on the survival analysis case and assume familiarity
#with the basic functionality of eval_design_mc.

#We first generate simple 2-level design using expand.grid:
basicdesign = expand.grid(a=c(-1, 1))
design = gen_design(candidateset=basicdesign, model=~a, trials=15)

#We can then evaluate the power of the design in the same way as eval_design_mc,
#now including the type of censoring (either right or left) and the point at which
#the data should be censored:

eval_design_survival_mc(RunMatrix=design, model=~a, alpha=0.05,
                       nsim=100, distribution="exponential",
                       censorpoint=5,censortype="right")

#Built-in Monte Carlo random generating functions are included for the gaussian, exponential,
#and lognormal distributions.

#We can also evaluate different censored distributions by specifying a custom
#random generating function and changing the distribution argument. You can also
#specify any additional arguments at the end of the function call and they will be
#input into the survreg function when it evaluates.

rlognorm = function(X, b) {
 Y = rlnorm(n=nrow(X), meanlog = X \%*\% b, sdlog = 0.4)
 censored = Y > 1.2
 Y[censored] = 1.2
 return(survival::Surv(time=Y, event=!censored, type="right"))
}

#Any additional arguments are passed into the survreg function call.  As an example, you
#might want to fix the "scale" argument to survreg, when fitting a lognormal:

eval_design_survival_mc(RunMatrix=design, model=~a, alpha=0.2, nsim=100,
                       distribution="lognormal", rfunctionsurv=rlognorm,
                       anticoef=c(0.184,0.101), delta=2, scale=0.4)
}
