% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evalGrids.R
\name{evalGrids}
\alias{evalGrids}
\title{Workhorse for simulation studies}
\usage{
evalGrids(dataGrid, procGrid = expandGrid(proc = "length"),
  replications = 1, discardGeneratedData = FALSE, progress = FALSE,
  summary.fun = NULL, ncpus = 1L, cluster = NULL,
  clusterSeed = rep(12345, 6), clusterLibraries = NULL,
  clusterGlobalObjects = NULL, fallback = NULL, envir = globalenv(), ...)
}
\arguments{
\item{dataGrid}{a \code{data.frame} where the first column
is a character vector with function names. The other
columns contain parameters for the functions specified
in the first column. Parameters with NA are ignored.}

\item{procGrid}{similar as \code{dataGrid} the first
column must contain function names. The other columns
contain parameters for the functions specified in the
first column. The data generated according to
\code{dataGrid} will always be passed to the first
unspecified argument of the functions specified in the first
column of \code{procGrid}.}

\item{replications}{number of replications for the simulation}

\item{discardGeneratedData}{if \code{TRUE} the generated
data is deleted after all function constellations in
\code{procGrid} have been applied. Otherwise, ALL
generated data sets will be part of the returned object.}

\item{progress}{if \code{TRUE} a progress bar is shown in the console.}

\item{summary.fun}{univariate functions to summarize the results (numeric or logical) over
the replications, e.g. mean, sd. Alternatively, \code{summary.fun} can be one
function that may return a vector.}

\item{ncpus}{a cluster of \code{ncpus} workers (R-processes)
is created on the local machine to conduct the
simulation. If \code{ncpus}
equals one no cluster is created and the simulation
is conducted by the current R-process.}

\item{cluster}{a cluster generated by the \code{parallel}
package that will be used to conduct the simulation.
If \code{cluster} is specified, then \code{ncpus} will
be ignored.}

\item{clusterSeed}{if the simulation is done in parallel
manner, then the combined multiple-recursive generator from L'Ecuyer (1999)
is used to generate random numbers. Thus \code{clusterSeed} must be a (signed) integer
vector of length 6. The 6 elements of the seed are internally regarded as
32-bit unsigned integers. Neither the first three nor the last three
should be all zero, and they are limited to less than 4294967087 and
4294944443 respectively.}

\item{clusterLibraries}{a character vector specifying
the packages that should be loaded by the workers.}

\item{clusterGlobalObjects}{a character vector specifying
the names of R objects in the global environment that should
be exported to the global environment of every worker.}

\item{fallback}{must be missing or a character specifying a file. Every time
when the data generation function is changed, the results so far obtained
are saved in the file specified by \code{fallback}.}

\item{envir}{must be provided if the functions specified
in \code{dataGrid} or \code{procGrid} are not part
of the global environment.}

\item{...}{only needed to alert the user if some deprecated arguments were used.}
}
\value{
The returned object is a list of the class
 \code{evalGrid}, where the fourth element is a list of lists named
 \code{simulation}. \code{simulation[[i]][[r]]} contains:
 \item{data}{the data set that was generated by the
 \code{i}th constellation (\code{i}th row) of \code{dataGrid}
 in the \code{r}th replication}
 \item{results}{a list containing \code{nrow(procGrid)}
 objects. The \code{j}th object is the returned value
 of the function specified by the \code{j}th constellation
 (\code{j}th row) of \code{procGrid} applied to the data
 set contained in \code{data}}
}
\description{
Generates data according to all provided
 constellations in \code{dataGrid} and applies
 all provided constellations in \code{procGrid}
 to them.
}
\note{
If \code{cluster} is provided by the user the
 function \code{evalGrids} will NOT stop the cluster.
 This has to be done by the user. Conducting parallel
 simulations by specifying \code{ncpus} will internally
 create a cluster and stop it after the simulation
 is done.
}
\examples{

rng = function(data, ...) {
ret = range(data)
names(ret) = c("min", "max")
ret
}

# call runif(n=1), runif(n=2), runif(n=3)
# and range on the three "datasets"
# generated by runif(n=1), runif(n=2), runif(n=3)
eg = evalGrids(
 expandGrid(fun="runif", n=1:3),
 expandGrid(proc="rng"),
 rep=10
)
eg

# summarizing the results in a data.frame
as.data.frame(eg)

# we now generate data for a regression
# and fit different regression models

# not that we use SD and not sd (the
# reason for this is the cast() call below)
regData = function(n, SD){
 data.frame(
   x=seq(0,1,length=n),
   y=rnorm(n, sd=SD))
}

eg = evalGrids(
 expandGrid(fun="regData", n=20, SD=1:2),
 expandGrid(proc="lm", formula=c("y~x", "y~I(x^2)")),
 replications=2)

# can not be converted to data.frame, because
# an object of class "lm" can not converted to
# a data.frame
try(as.data.frame(eg))

# for the data.frame we just extract the r.squared
# from the fitted model
as.data.frame(eg, convert.result.fun=function(fit) c(rsq=summary(fit)$r.squared))

# for the data.frame we just extract the coefficients
# from the fitted model
df = as.data.frame(eg, convert.result.fun=coef)

# since we have done 2 replication we can calculate
# sum summary statistics
library("reshape")
df$replication=NULL
mdf = melt(df, id=1:7, na.rm=TRUE)
cast(mdf, ... ~ ., c(mean, length, sd))

# note if the data.frame would contain the column
# named "sd" instead of "SD" the cast will generate
# an error
names(df)[5] = "sd"
mdf = melt(df, id=1:7, na.rm=TRUE)
try(cast(mdf, ... ~ ., c(mean, length, sd)))


# extracting the summary of the fitted.model
as.data.frame(eg, convert.result.fun=function(x) {
 ret = coef(summary(x))
 data.frame(valueName = rownames(ret), ret, check.names=FALSE)
})



# we now compare to methods for
# calculating quantiles

# the functions and parameters
# that generate the data
N = c(10, 50, 100)
library("plyr")
dg = rbind.fill(
 expandGrid(fun="rbeta", n=N, shape1=4, shape2=4),
 expandGrid(fun="rnorm", n=N))

# definition of the two quantile methods
emp.q = function(data, probs) c(quantile(data, probs=probs))
nor.q = function(data, probs) {
 ret = qnorm(probs, mean=mean(data), sd=sd(data))
 names(ret) = names(quantile(1, probs=probs))
 ret
}

# the functions and parameters that are
# applied to the generate data
pg = rbind.fill(expandGrid(proc=c("emp.q", "nor.q"), probs=c(0.01, 0.025, 0.05)))

# generate data and apply quantile methods
set.seed(1234)
eg = evalGrids(dg, pg, replication=50, progress=TRUE)

# convert the results to a data.frame
df = as.data.frame(eg)
df$replication=NULL
mdf = melt(df, id=1:8, na.rm=TRUE)

# calculate, print and plot summary statistics
require("ggplot2")
print(a <- arrange(cast(mdf, ... ~ ., c(mean, sd)), n))
ggplot(a, aes(x=fun, y=mean, color=proc)) + geom_point(size=I(3)) + facet_grid(probs ~ n)
}
\seealso{
\code{\link{as.data.frame.evalGrid}}
}
\author{
Marsel Scheer
}
