% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sglg2.R
\name{sglg}
\alias{sglg}
\title{Fitting semi-parametric generalized log-gamma regression models}
\usage{
sglg(formula, npc, basis, data, shape, method, Tolerance, Maxiter)
}
\arguments{
\item{formula}{a symbolic description of the systematic component of the model to be fitted. See details for further information.}

\item{npc}{a data frame with potential nonparametric variables of the systematic part of the model to be fitted.}

\item{basis}{a name of the cubic spline basis to be used in the model. Supported basis include deBoor and Gu basis
which are a B-spline basis and a natural cubic spline basis, respectively.}

\item{data}{an optional data frame, list containing the variables in the model.}

\item{shape}{an optional value for the shape parameter of the error distribution of a generalized log-gamma distribution. Default value is 1.}

\item{method}{There are two possibles algorithms to estimate the parameters. The default algorithm is 'FS' Fisher-Scoring,
the other option is 'GSFS' an adequate combination between the block matrix version of non-linear Gauss-Seidel algorithm and Fisher-Scoring algorithm.}

\item{Tolerance}{an optional positive value, which represents the convergence criterion. Default value is 1e-04.}

\item{Maxiter}{an optional positive integer giving the maximal number of iterations for the estimating process. Default value is 1e03.}
}
\value{
mu a vector of parameter estimates asociated with the location parameter.

sigma estimate of the scale parameter associated with the model.

lambda estimate of the shape parameter associated with the model.

interval estimate of a 95\% confidence interval for each estimate parameters associated with the model.

Deviance the deviance associated with the model.
}
\description{
\code{sglg} may be used to fit a semi-parametric regression model suitable for analysis of data sets in which the response variable is continuous, strictly positive, and asymmetric.
However, \code{sglg} may also be used to fit data for which the response is asymmetric in the real numbers. In this setup, the location parameter of the response variable is explicitly modeled by semi-parametric functions, whose nonparametric components may be approximated by
natural cubic splines or cubic P-splines.
}
\examples{
rows <- 175 # Number of observations
columns <- 2 # Number of parametric components
library(ssym)
t_beta  <- c(0.5, 2)
t_sigma <- 1
t_lambda <- 1
t_knot1 <- 7
ts1 <- seq(0, 1, length = t_knot1)
t_g1 <- 0.4 * sin(pi * ts1)

BasisN <- function(n, knot) {
          N <- matrix(0, n, knot)
          m <- n/knot
          block <- matrix(1, m, 1)
          for (i in 1:knot) {
          l <- (i - 1) * m + 1
          r <- i * m
          N[l:r, i] <- block }
          return(N)
          }
s_N1 <- BasisN(rows, length(ts1))
x3 <- s_N1 \%*\% ts1
colnames(x3) <- 'x3'
set.seed(8142031)
x1 <- rbinom(rows, 1, 0.5)
x2 <- runif(rows, 0, 1)
X <- cbind(x1,x2)
error <- rglg(rows, 0, 1, t_lambda)
y1 <- X \%*\%t_beta + + s_N1 \%*\% t_g1 + t_sigma * error
data.example <- data.frame(y1,X,x3)
fit1 <- sglg(y1 ~ x1 + x2 - 1, npc=x3, method='FS',data=data.example)
fit2 <- sglg(y1 ~ x1 + x2 - 1, npc=x3, method='GSFS',data=data.example)
\dontrun{

###########################################
#                                         #
# Diagnostic related grouping system, DRG #
#                                         #
###########################################

library(robustloggamma)
library(ssym)
data(drg2000)
attach(drg2000)
LOS2 <-split(LOS,factor(MDC))
MDC2 <-split(Cost,factor(MDC))
y <-log(MDC2$'3')
x <-as.matrix(LOS2$'3')
colnames(x) <- 'LOS'
Data <-data.frame(y,x)
fit1 <-glg(y~LOS,data=Data)
summary(fit1)
fit2 <-sglg(y~1,npc=x,data=Data)
summary(fit2)
plot(fit2)
}

}
\references{
Carlos A. Cardozo, G. Paula and L. Vanegas. Semi-parametric generalized log-gamma regression models. In preparation.

Carlos Alberto Cardozo Delgado, Semi-parametric generalized log-gamma regression models. Ph. D. thesis. Sao Paulo University.
}
\author{
Carlos Alberto Cardozo Delgado <cardozorpackages@gmail.com>, G. Paula and L. Vanegas.
}
