mufreq.dbinom <- function(mufreq, mufreq.model, depth.t, seq.errors = 0.01) {
   mufreq.model[mufreq.model == 0] <- seq.errors
   n.success       <- round(mufreq * depth.t, 0)
   dbinom( x = n.success, size = depth.t, prob = mufreq.model)
}

mufreq.dpois <- function(mufreq, mufreq.model, depth.t, seq.errors = 0.01) {
   mufreq.model[mufreq.model == 0] <- seq.errors
   n.success       <- round(mufreq * depth.t, 0)
   dpois( x = n.success, lambda = mufreq.model * depth.t)
}

baf.dbinom <- function(baf, baf.model, depth.t) {
   n.success       <- round(baf * depth.t, 0)
   dbinom( x = n.success, size = depth.t, prob = baf.model)
}

baf.dpois <- function(baf, baf.model, depth.t) {
   n.success       <- round(baf * depth.t, 0)
   dpois( x = n.success, lambda = baf.model * depth.t)
}

depth.ratio.dbinom <- function(size, depth.ratio, depth.ratio.model) {
   #n.success        <- round(depth.n * depth.ratio, 0)
   n.success        <- round(size * (depth.ratio/(1 + depth.ratio)), 0)
   prob             <- depth.ratio.model / (1 + depth.ratio.model)
   dbinom( x = n.success, size = size, prob = prob)
}

depth.ratio.dpois <- function(size, depth.ratio, depth.ratio.model) {
   #n.success        <- round(depth.n * depth.ratio, 0)
   n.success        <- round(size * (depth.ratio/(1 + depth.ratio)), 0)
   prob             <- depth.ratio.model / (1 + depth.ratio.model)
   dpois( x = n.success, lambda = prob * size)
}

mufreq.bayes <- function(mufreq, depth.ratio, cellularity, ploidy, avg.depth.ratio,
                         weight.mufreq = 100, weight.ratio = 100, CNt.min = 1, CNt.max = 7,
                         CNn = 2, priors.table = data.frame(CN = CNt.min:CNt.max, value = 1)) {

   mufreq.tab <- data.frame(F = mufreq, ratio = depth.ratio,
                            weight.mufreq = weight.mufreq, weight.ratio = weight.ratio)
   types <- types.matrix(CNt.min = CNt.min, CNt.max = CNt.max, CNn = CNn)
   mufreq.depth.ratio <- cbind(types, model.points(cellularity = cellularity, ploidy = ploidy,
                                                   types = types, avg.depth.ratio = avg.depth.ratio))
   rows.x             <- 1:nrow(mufreq.tab)

   priors <- rep(1, nrow(mufreq.depth.ratio))
   for (i in 1:nrow(priors.table)) {
      priors[mufreq.depth.ratio$CNt == priors.table$CN[i]] <- priors.table$value[i]
   }
   priors <- priors / sum(priors)

   bayes.fit <- function (x, mat, model.pts, priors) {
      test.ratio <- model.pts$depth.ratio
      test.mufrq <- model.pts$mufreqs
      min.offset <- 1e-323
      score.r    <- depth.ratio.dbinom(size = mat[x,]$weight.ratio, depth.ratio = mat[x,]$ratio, test.ratio)
      score.m    <- mufreq.dbinom(mufreq = mat[x,]$F, depth.t = mat[x,]$weight.mufreq, test.mufrq)

      score.r    <- score.r * priors
      score.m    <- score.m

      post.model <- score.r * score.m

      post.model[post.model == 0] <- min.offset
      # max.lik <-  which.max(post.model)
      # max.post <- c(as.numeric(model.pts[max.lik,1:3]), log2(post.model[max.lik]))
      # max.post

      res.cn     <- model.pts$CNt[which.max(score.r)]
      idx.pts    <- model.pts$CNt == res.cn
      model.lik  <- cbind(model.pts[idx.pts, 1:3], log2(post.model[idx.pts]))
      if (is.null(dim(model.lik))) {
         max.post <- model.lik
      } else {
         max.post   <- model.lik[which.max(model.lik[,4]),]
      }

      max.post
   }
   types.L           <- mapply(FUN = bayes.fit, rows.x,
                         MoreArgs = list(mat = mufreq.tab,
                                         model.pts = mufreq.depth.ratio,
                                         priors = priors),
                                         SIMPLIFY = FALSE)
   types.L           <- do.call(rbind, types.L)
   colnames(types.L) <- c("CNn","CNt","Mt", "L")
   types.L
}

baf.bayes <- function(Bf, depth.ratio, cellularity, ploidy, avg.depth.ratio,
                      weight.Bf = 100, weight.ratio = 100, CNt.min = 0,
                      CNt.max = 7, CNn = 2, priors.table = data.frame(CN = CNt.min:CNt.max,
                      value = 1), ratio.priority = FALSE, skew.baf = 0.95) {

   mufreq.tab <- data.frame(Bf = Bf, ratio = depth.ratio,
                            weight.Bf = weight.Bf, weight.ratio = weight.ratio)
   mufreq.depth.ratio <- model.points(cellularity = cellularity, ploidy = ploidy,
                                      types = cbind(CNn = CNn, CNt = CNt.min:CNt.max, Mt = 0),
                                      avg.depth.ratio = avg.depth.ratio)
   model.d.ratio      <- cbind(CNt = CNt.min:CNt.max, depth.ratio = mufreq.depth.ratio[, 2])
   model.baf          <- theoretical.baf(CNn = CNn, CNt = CNt.max, cellularity = cellularity)
   if(CNt.min == 0) {
      model.baf          <- as.data.frame(rbind(c(0, 0, 1/CNn, 0), model.baf))
   }
   # B-allele freq are never 0.5, always smaller. work around on this bias
   model.baf$BAF[model.baf$BAF == 0.5] <- quantile(rep(mufreq.tab$Bf, times = mufreq.tab$weight.Bf),
                                                   na.rm = TRUE, probs = skew.baf)
   model.pts          <- merge(model.baf, model.d.ratio)
   # model.pts          <- cbind(baf.type = apply(model.pts[, 1:3], 1, FUN = function(x) paste(x, collapse = "_")),
   #                             model.pts[, 4:5])
   rows.x             <- 1:nrow(mufreq.tab)

   priors <- rep(1, nrow(model.pts))
   for (i in 1:nrow(priors.table)) {
      priors[model.pts$CNt == priors.table$CN[i]] <- priors.table$value[i]
   }
   priors <- priors / sum(priors)

   bayes.fit <- function (x, mat, model.pts, priors, ratio.priority) {
      test.ratio <- model.pts$depth.ratio
      test.baf   <- model.pts$BAF
      min.offset <- 1e-323
      score.r    <- depth.ratio.dbinom(size = mat[x,]$weight.ratio, depth.ratio = mat[x,]$ratio, test.ratio)
      score.b    <- baf.dbinom(baf = mat[x,]$Bf, depth.t = mat[x,]$weight.Bf, test.baf)

      score.r    <- score.r * priors
      score.b    <- score.b

      post.model <- score.r * score.b

      post.model[post.model == 0] <- min.offset
      if (ratio.priority == FALSE) {
         max.lik <-  which.max(post.model)
         max.post <- c(as.numeric(model.pts[max.lik,1:3]), log2(post.model[max.lik]))
      } else {
         res.cn     <- model.pts$CNt[which.max(score.r)]
         idx.pts    <- model.pts$CNt == res.cn
         model.lik  <- cbind(model.pts[idx.pts, 1:3], log2(post.model[idx.pts]))
         if (is.null(dim(model.lik))) {
            max.post <- model.lik
         } else {
            max.post   <- model.lik[which.max(model.lik[,4]),]
         }
      }
      max.post
   }
   bafs.L           <- mapply(FUN = bayes.fit, rows.x,
                         MoreArgs = list(mat = mufreq.tab,
                                         model.pts = model.pts,
                                         priors = priors,
                                         ratio.priority = ratio.priority),
                                         SIMPLIFY = FALSE)
   bafs.L           <- do.call(rbind, bafs.L)
   colnames(bafs.L) <- c("CNt", "A", "B", "L")
   bafs.L
}

shannon.types <- function(types.mat) {
   data.types <- apply(types.mat, 1, FUN = function(x) paste(x, collapse = "_"))
   tab.types <- table(data.types)/length(data.types)
   tab.types <- tab.types * log2(tab.types)
   -sum(tab.types)
}

mufreq.model.fit <- function(cellularity = seq(0.3, 1, by = 0.01),
                          ploidy = seq(1, 7, by = 0.1),
                          mc.cores = getOption("mc.cores", 2L), ...) {

   result <- expand.grid(ploidy = ploidy, cellularity = cellularity,
                         KEEP.OUT.ATTRS = FALSE)

   fit.cp <- function(ii) {
      L.model <- mufreq.bayes(cellularity = result$cellularity[ii],
                           ploidy = result$ploidy[ii], ...)
      sum(L.model[,4])
   }
   bayes.res <- mclapplyPb(X = 1:nrow(result), FUN = fit.cp, mc.cores = mc.cores)
   result$L <- unlist(bayes.res)
   z <- tapply(result$L, list(result$ploidy, result$cellularity), mean)
   x <- as.numeric(rownames(z))
   y <- as.numeric(colnames(z))
   max.lik <- max(result$L)
   LogSumLik <- log2(sum(2^(result$L - max.lik))) + max.lik
   znorm <- 2^(z - LogSumLik)
   list(x = x, y = y, z = znorm)
}

baf.model.fit <- function(cellularity = seq(0.3, 1, by = 0.01),
                          ploidy = seq(1, 7, by = 0.1),
                          mc.cores = getOption("mc.cores", 2L), ...) {

   result <- expand.grid(ploidy = ploidy, cellularity = cellularity,
                         KEEP.OUT.ATTRS = FALSE)

   fit.cp <- function(ii) {
      L.model <- baf.bayes(cellularity = result$cellularity[ii],
                           ploidy = result$ploidy[ii], ...)
      sum(L.model[,4])
   }
   bayes.res <- mclapplyPb(X = 1:nrow(result), FUN = fit.cp, mc.cores = mc.cores)
   result$L <- unlist(bayes.res)
   z <- tapply(result$L, list(result$ploidy, result$cellularity), mean)
   x <- as.numeric(rownames(z))
   y <- as.numeric(colnames(z))
   max.lik <- max(result$L)
   LogSumLik <- log2(sum(2^(result$L - max.lik))) + max.lik
   znorm <- 2^(z - LogSumLik)
   list(x = x, y = y, z = znorm)
}
