% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cross-val.R
\name{sdmTMB_cv}
\alias{sdmTMB_cv}
\title{Cross validation with sdmTMB models}
\usage{
sdmTMB_cv(
  formula,
  data,
  mesh_args,
  mesh = NULL,
  time = NULL,
  k_folds = 8,
  fold_ids = NULL,
  lfo = FALSE,
  lfo_forecast = 1,
  lfo_validations = 5,
  parallel = TRUE,
  use_initial_fit = FALSE,
  save_models = TRUE,
  future_globals = NULL,
  ...
)
}
\arguments{
\item{formula}{Model formula.}

\item{data}{A data frame.}

\item{mesh_args}{Arguments for \code{\link[=make_mesh]{make_mesh()}}. If supplied, the mesh will be
reconstructed for each fold.}

\item{mesh}{Output from \code{\link[=make_mesh]{make_mesh()}}. If supplied, the same mesh will be
used for all folds. This is faster and usually what you want.}

\item{time}{The name of the time column. Leave as \code{NULL} if this is only
spatial data.}

\item{k_folds}{Number of folds.}

\item{fold_ids}{Optional vector containing user fold IDs. Can also be a
single string, e.g. \code{"fold_id"} representing the name of the variable in
\code{data}. Ignored if \code{lfo} is TRUE}

\item{lfo}{Logical. Use leave-future-out (LFO) cross validation? If \code{TRUE},
data from earlier time steps are used to predict future time steps. The
\code{time} argument must be specified. See Details section below.}

\item{lfo_forecast}{If \code{lfo = TRUE}, number of time steps ahead to forecast.
For example, \code{lfo_forecast = 1} means fitting to time steps 1 to T and
validating on T + 1. See Details section below.}

\item{lfo_validations}{If \code{lfo = TRUE}, number of times to step through the
LFO process (i.e., number of validation folds). Defaults to 5. See Details
section below.}

\item{parallel}{If \code{TRUE} and a \code{\link[future:plan]{future::plan()}} is supplied, will be run in
parallel.}

\item{use_initial_fit}{Fit the first fold and use those parameter values
as starting values for subsequent folds? Can be faster with many folds.}

\item{save_models}{Logical. If \code{TRUE} (default), the fitted model object for
each fold is stored in the output. If \code{FALSE}, models are not saved, which
can substantially reduce memory usage for large datasets or many folds.
When \code{FALSE}, functions that require access to the fitted models (e.g.,
\code{\link[=tidy]{tidy()}}, \code{\link[=cv_to_waywiser]{cv_to_waywiser()}}) will not work.}

\item{future_globals}{A character vector of global variables used within
arguments if an error is returned that \pkg{future.apply} can't find an
object. This vector is appended to \code{TRUE} and passed to the argument
\code{future.globals} in \code{\link[future.apply:future_lapply]{future.apply::future_lapply()}}. Useful if global
objects are used to specify arguments like priors, families, etc.}

\item{...}{All other arguments required to run the \code{\link[=sdmTMB]{sdmTMB()}} model. The
\code{weights} argument is supported and will be combined with the internal
fold-assignment mechanism (held-out data are assigned weight 0).}
}
\value{
A list:
\itemize{
\item \code{data}: Original data plus columns for fold ID (\code{cv_fold}), CV predicted
value (\code{cv_predicted}), CV log likelihood (\code{cv_loglik}), and CV deviance
residuals (\code{cv_deviance_resid}).
\item \code{models}: A list of fitted models, one per fold. \code{NULL} if \code{save_models = FALSE}.
\item \code{fold_loglik}: Sum of log likelihoods of held-out data per fold (log
predictive density per fold). More positive values indicate better
out-of-sample prediction.
\item \code{sum_loglik}: Sum of \code{fold_loglik} across all folds (total log predictive
density). Use this to compare models—more positive values are better.
\item \code{pdHess}: Logical vector: was the Hessian positive definite for each fold?
\item \code{converged}: Logical: did all folds converge (all \code{pdHess} \code{TRUE})?
\item \code{max_gradients}: Maximum absolute gradient for each fold.
}
}
\description{
Performs k-fold or leave-future-out cross validation with sdmTMB models.
Returns the sum of log likelihoods of held-out data (log predictive density),
which can be used to compare models—higher values indicate better
out-of-sample prediction. By default, creates folds randomly and stratified
by time (set a seed for reproducibility), but folds can be manually assigned
via \code{fold_ids}. See Ward and Anderson (2025) in the References and the
\href{https://sdmTMB.github.io/sdmTMB/articles/cross-validation.html}{cross-validation vignette}.
}
\details{
\strong{Parallel processing}

Parallel processing can be used by setting a \code{future::plan()}.

For example:

\if{html}{\out{<div class="sourceCode">}}\preformatted{library(future)
plan(multisession)
# now use sdmTMB_cv() ...
}\if{html}{\out{</div>}}

\strong{Leave-future-out cross validation (LFOCV)}

An example of LFOCV with 9 time steps, \code{lfo_forecast = 1}, and
\code{lfo_validations = 2}:
\itemize{
\item Fit data to time steps 1 to 7, predict and validate step 8.
\item Fit data to time steps 1 to 8, predict and validate step 9.
}

An example of LFOCV with 9 time steps, \code{lfo_forecast = 2}, and
\code{lfo_validations = 3}:
\itemize{
\item Fit data to time steps 1 to 5, predict and validate step 7.
\item Fit data to time steps 1 to 6, predict and validate step 8.
\item Fit data to time steps 1 to 7, predict and validate step 9.
}

Note these are time steps as they are presented in order in the data.
For example, in the \code{pcod} data example below steps between data points
are not always one year but an \code{lfo_forecast = 2} forecasts 2 time
steps as presented not two years.

See example below.
}
\examples{
mesh <- make_mesh(pcod, c("X", "Y"), cutoff = 25)

# Set parallel processing first if desired with the future package.
# See the Details section above.

m_cv <- sdmTMB_cv(
  density ~ 0 + depth_scaled + depth_scaled2,
  data = pcod, mesh = mesh, spatial = "off",
  family = tweedie(link = "log"), k_folds = 2
)

m_cv$fold_loglik
m_cv$sum_loglik

head(m_cv$data)
m_cv$models[[1]]
m_cv$max_gradients


\donttest{
# Create mesh each fold:
m_cv2 <- sdmTMB_cv(
  density ~ 0 + depth_scaled + depth_scaled2,
  data = pcod, mesh_args = list(xy_cols = c("X", "Y"), cutoff = 20),
  family = tweedie(link = "log"), k_folds = 2
)

# Use fold_ids:
m_cv3 <- sdmTMB_cv(
  density ~ 0 + depth_scaled + depth_scaled2,
  data = pcod, mesh = mesh,
  family = tweedie(link = "log"),
  fold_ids = rep(seq(1, 3), nrow(pcod))[seq(1, nrow(pcod))]
)

# LFOCV:
m_lfocv <- sdmTMB_cv(
  present ~ s(year, k = 4),
  data = pcod,
  lfo = TRUE,
  lfo_forecast = 2,
  lfo_validations = 3,
  family = binomial(),
  mesh = mesh,
  spatial = "off", # fast example
  spatiotemporal = "off", # fast example
  time = "year" # must be specified
)

# See how the LFOCV folds were assigned:
fold_table <- table(m_lfocv$data$cv_fold, m_lfocv$data$year)
fold_table
}
}
\references{
Ward, E.J., and S.C. Anderson. 2025. Approximating spatial processes with
too many knots degrades the quality of probabilistic predictions.
bioRxiv 2025.11.14.688354. \doi{10.1101/2025.11.14.688354}.
}
