% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/options.R
\name{sass_cache_options}
\alias{sass_cache_options}
\title{Caching Options for Sass}
\usage{
sass_cache_options(
  cache = getOption("sass.cache", !interactive()),
  cache_dir = getOption("sass.cache_dir", tempdir())
)
}
\arguments{
\item{cache}{Logical value indicating whether caching is performed. If no
value is provided, the R option \code{sass.cache} is consulted; if the
option is not set, then caching is performed only if the R session is not
\code{\link{interactive}}.}

\item{cache_dir}{The directory which will be used to store cache files. If no
value is provided, the R option \code{sass.cache_dir} is consulted; if the
option is not set, then \code{\link{tempdir}} is used. Note that this means
that caches will not persist beyond the current R session by default, since
\code{tempdir()} is unique to each session.}
}
\description{
Specifies whether caching is used with sass, and where on the file system the
cached data should be stored. Used with \code{\link{sass}}.
}
\details{
If caching is enabled, \code{sass()} will attempt to bypass the compilation
process by reusing output from previous \code{sass()} calls that used
equivalent inputs. This mechanism works by using a hashing algorithm to
derive a \emph{cache key} from each \code{sass()} call's \code{input} and
\code{options} arguments. If a file named
\emph{CACHE_KEY}\code{.sasscache.css} exists within the cache directory, its
contents are used instead of performing the compilation. If the file does not
exist, then compilation is performed and usual and the results are stored at
that file path for next time.

If a file that is included using \code{\link{sass_file}} changes on disk
(i.e. its last-modified time changes), its previous cache entries will
effectively be invalidated (not removed from disk, but they'll no longer be
matched). However, if a file imported using \code{sass_file} itself imports
other sass files using \code{@import}, changes to those files are invisible
to the cache and you will end up with stale results.

If a cache directory is explicitly specified (either via the \code{cache_dir}
argument or via the \code{sass.cache_dir} R option), note that this package
does not do any cleanup of that directory. If disk space is a concern, you
will need to delete older entries from that directory yourself.
}
\examples{
# Very slow to compile
fib_sass <- "@function fib($x) {
  @if $x <= 1 {
    @return $x
  }
  @return fib($x - 2) + fib($x - 1);
}

body {
  width: fib(27);
}"

# Use a custom cache dir for the purposes of this example. Normally,
# you'd want to set the caching behavior using options().
temp_cache_dir <- tempfile("sass_cache_dir")
dir.create(temp_cache_dir)
cache_opts <- sass_cache_options(TRUE, cache_dir = temp_cache_dir)

# The first time this runs it will be very slow
system.time(sass(fib_sass, cache_options = cache_opts))

# But on subsequent calls, it should be very fast
system.time(sass(fib_sass, cache_options = cache_opts))

}
