source('sanitizeCovr.R')

test_that('1 rule makefile', {
  job <- list(rule(target='target.Rdata', depends=c('dep1', 'dep2'), build='buildCmd', clean='cleanCmd'))
  res <- makefile(job, fileName=NULL)

  res <- sanitizeCovr(res)
  res <- res[-seq_len(which(res == ''))]
  expect_equal(res, c('.PHONY: all',
                      'all: target.Rdata',
                      '\t',
                      'target.Rdata: dep1 dep2',
                      '\tbuildCmd',
                      '.PHONY: clean',
                      'clean: ',
                      '\tcleanCmd',
                      'Makefile: Makefile.R',
                      '\t$(R) -e \'{\' \\',
                      '\t-e \'    params <- NULL\' \\',
                      '\t-e \'    source("Makefile.R")\' \\',
                      '\t-e \'}\''))
})

test_that('not all in makefile', {
  job <- list(rule(target='target.Rdata', depends=c('dep1', 'dep2'), build='buildCmd', clean='cleanCmd'))
  res <- makefile(job, fileName=NULL, all=FALSE)
  res <- sanitizeCovr(res)
  res <- res[-seq_len(which(res == ''))]
  expect_equal(res, c('target.Rdata: dep1 dep2',
                      '\tbuildCmd',
                      '.PHONY: clean',
                      'clean: ',
                      '\tcleanCmd',
                      'Makefile: Makefile.R',
                      '\t$(R) -e \'{\' \\',
                      '\t-e \'    params <- NULL\' \\',
                      '\t-e \'    source("Makefile.R")\' \\',
                      '\t-e \'}\''))
})

test_that('not clean in makefile', {
  job <- list(rule(target='target.Rdata', depends=c('dep1', 'dep2'), build='buildCmd', clean='cleanCmd'))
  res <- makefile(job, fileName=NULL, clean=FALSE)
  res <- sanitizeCovr(res)
  res <- res[-seq_len(which(res == ''))]
  expect_equal(res, c('.PHONY: all',
                      'all: target.Rdata',
                      '\t',
                      'target.Rdata: dep1 dep2',
                      '\tbuildCmd',
                      'Makefile: Makefile.R',
                      '\t$(R) -e \'{\' \\',
                      '\t-e \'    params <- NULL\' \\',
                      '\t-e \'    source("Makefile.R")\' \\',
                      '\t-e \'}\''))
})

test_that('not Makefile in makefile', {
  job <- list(rule(target='target.Rdata', depends=c('dep1', 'dep2'), build='buildCmd', clean='cleanCmd'))
  res <- makefile(job, fileName=NULL, makefile=FALSE)
  res <- sanitizeCovr(res)
  res <- res[-seq_len(which(res == ''))]
  expect_equal(res, c('.PHONY: all',
                      'all: target.Rdata',
                      '\t',
                      'target.Rdata: dep1 dep2',
                      '\tbuildCmd',
                      '.PHONY: clean',
                      'clean: ',
                      '\tcleanCmd'))
})

test_that('custom vars in makefile', {
  job <- list(rule(target='target.Rdata', depends=c('dep1', 'dep2'), build='buildCmd', clean='cleanCmd'))
  res <- makefile(job, fileName=NULL, vars=c(CPP='gcc'))
  res <- sanitizeCovr(res)
  res <- res[seq_len(which(res == ''))]
  if (.Platform$OS.type == 'unix') {
    expect_equal(res, c('# Generated by rmake: do not edit by hand',
                        'SHELL=/bin/sh',
                        'R="$(R_HOME)/bin$(R_ARCH)/Rscript"',
                        'RM=rm',
                        'CPP=gcc',
                        ''))
  } else {
    expect_equal(res, c('# Generated by rmake: do not edit by hand',
                        'SHELL=/bin/sh',
                        'R="$(R_HOME)/bin$(R_ARCH)/Rscript"',
                        'RM=del',
                        'CPP=gcc',
                        ''))
  }
})


test_that('redefined vars in makefile', {
  job <- list(rule(target='target.Rdata', depends=c('dep1', 'dep2'), build='buildCmd', clean='cleanCmd'))
  res <- makefile(job, fileName=NULL, vars=c(R='MyR', CPP='gcc'))
  res <- sanitizeCovr(res)
  res <- res[seq_len(which(res == ''))]
  if (.Platform$OS.type == 'unix') {
    expect_equal(res, c('# Generated by rmake: do not edit by hand',
                        'SHELL=/bin/sh',
                        'R=MyR',
                        'RM=rm',
                        'CPP=gcc',
                        ''))
  } else {
    expect_equal(res, c('# Generated by rmake: do not edit by hand',
                        'SHELL=/bin/sh',
                        'R=MyR',
                        'RM=del',
                        'CPP=gcc',
                        ''))
  }
})

test_that('multiple target rule makefile', {
  job <- list(rule(target=c('target.pdf', 'target.docx'),
                     depends=c('dep1', 'dep2'),
                     build='buildCmd',
                     clean='cleanCmd'))
  res <- makefile(job, fileName=NULL)
  res <- sanitizeCovr(res)
  res <- res[-seq_len(which(res == ''))]
  expect_equal(res, c('.PHONY: all',
                      'all: target.pdf target.docx',
                      '\t',
                      'target%pdf target%docx: dep1 dep2',
                      '\tbuildCmd',
                      '.PHONY: clean',
                      'clean: ',
                      '\tcleanCmd',
                      'Makefile: Makefile.R',
                      '\t$(R) -e \'{\' \\',
                      '\t-e \'    params <- NULL\' \\',
                      '\t-e \'    source("Makefile.R")\' \\',
                      '\t-e \'}\''))
})

test_that('makefile with tasks', {
  job <- list(rule(target=c('target1.pdf', 'target1.docx'),
                     depends=c('dep1.1', 'dep1.2'),
                     build='buildCmd1',
                     clean='cleanCmd1',
                     task='task1'),
              rule(target=c('target2.pdf', 'target2.docx'),
                     depends=c('dep2.1', 'dep2.2'),
                     build='buildCmd2',
                     clean='cleanCmd2',
                     task='task2'),
              rule(target=c('target3.pdf', 'target3.docx'),
                     depends=c('dep3.1', 'dep3.2'),
                     build='buildCmd3',
                     clean='cleanCmd3'))
  res <- makefile(job, fileName=NULL)
  res <- sanitizeCovr(res)
  res <- res[-seq_len(which(res == ''))]
  expect_equal(res, c('.PHONY: all',
                      'all: task1 task2 target3.pdf target3.docx',
                      '\t',
                      '.PHONY: task1',
                      'task1: target1.pdf target1.docx',
                      '\t',
                      '.PHONY: task2',
                      'task2: target2.pdf target2.docx',
                      '\t',
                      'target1%pdf target1%docx: dep1.1 dep1.2',
                      '\tbuildCmd1',
                      'target2%pdf target2%docx: dep2.1 dep2.2',
                      '\tbuildCmd2',
                      'target3%pdf target3%docx: dep3.1 dep3.2',
                      '\tbuildCmd3',
                      '.PHONY: clean',
                      'clean: ',
                      '\tcleanCmd1',
                      '\tcleanCmd2',
                      '\tcleanCmd3',
                      'Makefile: Makefile.R',
                      '\t$(R) -e \'{\' \\',
                      '\t-e \'    params <- NULL\' \\',
                      '\t-e \'    source("Makefile.R")\' \\',
                      '\t-e \'}\''))
})
