% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eval-tidy.R
\name{eval_tidy}
\alias{eval_tidy}
\title{Evaluate an expression with quosures and pronoun support}
\usage{
eval_tidy(expr, data = NULL, env = caller_env())
}
\arguments{
\item{expr}{An expression or quosure to evaluate.}

\item{data}{A data frame, or named list or vector. Alternatively, a
data mask created with \code{\link[=as_data_mask]{as_data_mask()}} or
\code{\link[=new_data_mask]{new_data_mask()}}. Objects in \code{data} have priority over those in
\code{env}. See the section about data masking.}

\item{env}{The environment in which to evaluate \code{expr}. This
environment is not applicable for quosures because they have
their own environments.}
}
\description{
\Sexpr[results=rd, stage=render]{rlang:::lifecycle("stable")}

\code{eval_tidy()} is a variant of \code{\link[base:eval]{base::eval()}} that powers the tidy
evaluation framework. Like \code{eval()} it accepts user data as
argument. Whereas \code{eval()} simply transforms the data to an
environment, \code{eval_tidy()} transforms it to a \strong{data mask} with
\code{\link[=as_data_mask]{as_data_mask()}}. Evaluating in a data mask enables the following
features:
\itemize{
\item \link[=nse-defuse]{Quosures}. Quosures are expressions bundled with an
environment. If \code{data} is supplied, objects in the data mask
always have precedence over the quosure environment, i.e. the
data masks the environment.
\item \link[=.data]{Pronouns}. If \code{data} is supplied, the \code{.env} and \code{.data}
pronouns are installed in the data mask. \code{.env} is a reference to
the calling environment and \code{.data} refers to the \code{data} argument.
These pronouns lets you be explicit about where to find
values and throw errors if you try to access non-existent values.
}
}
\section{Data masking}{


Data masking refers to how columns or objects inside \code{data} have
priority over objects defined in \code{env} (or in the quosure
environment, if applicable). If there is a column \code{var} in \code{data}
and an object \code{var} in \code{env}, and \code{expr} refers to \code{var}, the
column has priority:\preformatted{var <- "this one?"
data <- data.frame(var = rep("Or that one?", 3))

within <- function(data, expr) \{
  eval_tidy(enquo(expr), data)
\}

within(data, toupper(var))
#> [1] "OR THAT ONE?" "OR THAT ONE?" "OR THAT ONE?"
}

Because the columns or objects in \code{data} are always found first,
before objects from \code{env}, we say that the data "masks" the
environment.
}

\section{When should eval_tidy() be used instead of eval()?}{


\code{base::eval()} is sufficient for simple evaluation. Use
\code{eval_tidy()} when you'd like to support expressions referring to
the \code{.data} pronoun, or when you need to support quosures.

If you're evaluating an expression captured with quasiquotation
support, it is recommended to use \code{eval_tidy()} because users will
likely unquote quosures.

Note that unwrapping a quosure with \code{\link[=quo_get_expr]{quo_get_expr()}} does not
guarantee that there is no quosures inside the expression. Quosures
might be unquoted anywhere. For instance, the following does not
work reliably in the presence of nested quosures:\preformatted{my_quoting_fn <- function(x) \{
  x <- enquo(x)
  expr <- quo_get_expr(x)
  env <- quo_get_env(x)
  eval(expr, env)
\}

# Works:
my_quoting_fn(toupper(letters))

# Fails because of a nested quosure:
my_quoting_fn(toupper(!!quo(letters)))
}
}

\section{Stack semantics of \code{eval_tidy()}}{


\code{eval_tidy()} has different stack semantics than \code{\link[base:eval]{base::eval()}}:
\itemize{
\item Functions that require the evaluation environment to correspond
to a frame on the call stack do not work. This is why \code{return()}
called from a quosure does not work.
\item The mask environment creates a new branch in the call tree
defined by \code{sys.parent()} (you can visualise it in a \code{\link[=browser]{browser()}}
session with \code{lobstr::cst()}).
}

See also \code{\link[=eval_bare]{eval_bare()}} for more information about these differences.
}

\section{Life cycle}{


\strong{rlang 0.3.0}

Passing an environment to \code{data} is deprecated. Please construct an
rlang data mask with \code{\link[=new_data_mask]{new_data_mask()}}.
}

\examples{

# With simple quoted expressions eval_tidy() works the same way as
# eval():
apple <- "apple"
kiwi <- "kiwi"
expr <- quote(paste(apple, kiwi))
expr

eval(expr)
eval_tidy(expr)

# Both accept a data mask as argument:
data <- list(apple = "CARROT", kiwi = "TOMATO")
eval(expr, data)
eval_tidy(expr, data)


# In addition eval_tidy() has support for quosures:
with_data <- function(data, expr) {
  quo <- enquo(expr)
  eval_tidy(quo, data)
}
with_data(NULL, apple)
with_data(data, apple)
with_data(data, list(apple, kiwi))

# Secondly eval_tidy() installs handy pronouns that allow users to
# be explicit about where to find symbols:
with_data(data, .data$apple)
with_data(data, .env$apple)


# Note that instead of using `.env` it is often equivalent and may
# be preferred to unquote a value. There are two differences. First
# unquoting happens earlier, when the quosure is created. Secondly,
# subsetting `.env` with the `$` operator may be brittle because
# `$` does not look through the parents of the environment.
#
# For instance using `.env$name` in a magrittr pipeline is an
# instance where this poses problem, because the magrittr pipe
# currently (as of v1.5.0) evaluates its operands in a *child* of
# the current environment (this child environment is where it
# defines the pronoun `.`).
\dontrun{
  data \%>\% with_data(!!kiwi)     # "kiwi"
  data \%>\% with_data(.env$kiwi)  # NULL
}
}
\seealso{
\link{nse-force} for the second leg of the tidy evaluation
framework.
}
