% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cnd-abort.R, R/cnd-signal.R
\name{abort}
\alias{abort}
\alias{warn}
\alias{inform}
\alias{signal}
\alias{interrupt}
\title{Signal an error, warning, or message}
\usage{
abort(
  message = NULL,
  class = NULL,
  ...,
  trace = NULL,
  parent = NULL,
  .subclass
)

warn(
  message = NULL,
  class = NULL,
  ...,
  .frequency = c("always", "regularly", "once"),
  .frequency_id = NULL,
  .subclass
)

inform(
  message = NULL,
  class = NULL,
  ...,
  .file = NULL,
  .frequency = c("always", "regularly", "once"),
  .frequency_id = NULL,
  .subclass
)

signal(message, class, ..., .subclass)

interrupt()
}
\arguments{
\item{message}{The message to display.

If not supplied, it is expected that the message is generated
lazily through \link[=cnd_message]{conditionMessage()}. In that case,
\code{class} must be supplied. Only \code{inform()} allows empty messages
as it is occasionally useful to build user output incrementally.}

\item{class}{Subclass of the condition. This allows your users
to selectively handle the conditions signalled by your functions.}

\item{...}{Additional data to be stored in the condition object.}

\item{trace}{A \code{trace} object created by \code{\link[=trace_back]{trace_back()}}.}

\item{parent}{A parent condition object created by \code{\link[=abort]{abort()}}.}

\item{.subclass}{This argument was renamed to \code{class} in rlang
0.4.2.  It will be deprecated in the next major version. This is
for consistency with our conventions for class constructors
documented in \url{https://adv-r.hadley.nz/s3.html#s3-subclassing}.}

\item{.frequency}{How frequently should the warning or message be
displayed? By default (\code{"always"}) it is displayed at each
time. If \code{"regularly"}, it is displayed once every 8 hours. If
\code{"once"}, it is displayed once per session.}

\item{.frequency_id}{A unique identifier for the warning or
message. This is used when \code{.frequency} is supplied to recognise
recurring conditions. This argument must be supplied if
\code{.frequency} is not set to \code{"always"}.}

\item{.file}{Where the message is printed. This should be a
connection or character string which will be passed to \code{\link[=cat]{cat()}}.

By default, \code{inform()} prints to standard output in interactive
sessions and standard error otherwise. This way IDEs can treat
messages distinctly from warnings and errors, and R scripts can
still filter out the messages easily by redirecting \code{stderr}.}
}
\description{
These functions are equivalent to base functions \code{\link[base:stop]{base::stop()}},
\code{\link[base:warning]{base::warning()}}, and \code{\link[base:message]{base::message()}}, but make it easy to supply
condition metadata:
\itemize{
\item Supply \code{class} to create a classed condition. Typed
conditions can be captured or handled selectively, allowing for
finer-grained error handling.
\item Supply metadata with named \code{...} arguments. This data will be
stored in the condition object and can be examined by handlers.
}

\code{interrupt()} allows R code to simulate a user interrupt of the
kind that is signalled with \code{Ctrl-C}. It is currently not possible
to create custom interrupt condition objects.
}
\section{Backtrace}{


Unlike \code{stop()} and \code{warning()}, these functions don't include call
information by default. This saves you from typing \code{call. = FALSE}
and produces cleaner error messages.

A backtrace is always saved into error objects. You can print a
simplified backtrace of the last error by calling \code{\link[=last_error]{last_error()}}
and a full backtrace with \code{summary(last_error())}.

You can also display a backtrace with the error message by setting
the option \code{\link{rlang_backtrace_on_error}}. It supports the following
values:
\itemize{
\item \code{"reminder"}: Invite users to call \code{rlang::last_error()} to see a
backtrace.
\item \code{"branch"}: Display a simplified backtrace.
\item \code{"collapse"}: Display a collapsed backtrace tree.
\item \code{"full"}: Display a full backtrace tree.
\item \code{"none"}: Display nothing.
}
}

\section{Mufflable conditions}{


Signalling a condition with \code{inform()} or \code{warn()} causes a message
to be displayed in the console. These messages can be muffled with
\code{\link[base:message]{base::suppressMessages()}} or \code{\link[base:warning]{base::suppressWarnings()}}.

On recent R versions (>= R 3.5.0), interrupts are typically
signalled with a \code{"resume"} restart. This is however not
guaranteed.
}

\examples{
# These examples are guarded to avoid throwing errors
if (FALSE) {

# Signal an error with a message just like stop():
abort("Something bad happened")

# Give a class to the error:
abort("Something bad happened", "somepkg_bad_error")

# This will allow your users to handle the error selectively
tryCatch(
  somepkg_function(),
  somepkg_bad_error = function(err) {
    warn(conditionMessage(err)) # Demote the error to a warning
    NA                          # Return an alternative value
  }
)

# You can also specify metadata that will be stored in the condition:
abort("Something bad happened", "somepkg_bad_error", data = 1:10)

# This data can then be consulted by user handlers:
tryCatch(
  somepkg_function(),
  somepkg_bad_error = function(err) {
    # Compute an alternative return value with the data:
    recover_error(err$data)
  }
)

# If you call low-level APIs it is good practice to handle
# technical errors and rethrow them with a more meaningful
# message. Always prefer doing this from `withCallinghandlers()`
# rather than `tryCatch()` because the former preserves the stack
# on error and makes it possible for users to use `recover()`.
file <- "http://foo.bar/baz"
try(withCallinghandlers(
  download(file),
  error = function(err) {
    msg <- sprintf("Can't download `\%s`", file)
    abort(msg, parent = err)
}))
# Note how we supplied the parent error to `abort()` as `parent` to
# get a decomposition of error messages across error contexts.

# Unhandled errors are saved automatically by `abort()` and can be
# retrieved with `last_error()`. The error prints with a simplified
# backtrace:
abort("Saved error?")
last_error()

# Use `summary()` to print the full backtrace and the condition fields:
summary(last_error())

}
}
\seealso{
\code{\link[=with_abort]{with_abort()}} to convert all errors to rlang errors.
}
