% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nse-force.R
\name{nse-force}
\alias{nse-force}
\alias{quasiquotation}
\alias{UQ}
\alias{UQS}
\alias{{{}}}
\alias{\{\{}
\alias{!!}
\alias{!!!}
\alias{:=}
\alias{qq_show}
\title{Force parts of an expression}
\usage{
qq_show(expr)
}
\arguments{
\item{expr}{An expression to be quasiquoted.}
}
\description{
It is sometimes useful to force early evaluation of part of an
expression before it gets fully evaluated. The tidy eval framework
provides several forcing operators for different use cases.
\itemize{
\item The bang-bang operator \code{!!} forces a \emph{single} object. One
common case for \code{!!} is to substitute an environment-variable
(created with \code{<-}) with a data-variable (inside a data frame).\preformatted{library(dplyr)

# The environment variable `var` refers to the data-variable
# `height`
var <- sym("height")

# We force `var`, which substitutes it with `height`
starwars \%>\%
  summarise(avg = mean(!!var, na.rm = TRUE))
}
\item The big-bang operator \code{!!!} forces-splice a \emph{list} of objects.
The elements of the list are spliced in place, meaning that they
each become one single argument.\preformatted{vars <- syms(c("height", "mass"))

# Force-splicing is equivalent to supplying the elements separately
starwars \%>\% select(!!!vars)
starwars \%>\% select(height, mass)
}
\item The curly-curly operator \code{{{ }}} for function arguments is a bit
special because it forces the function argument and immediately
defuses it. The defused expression is substituted in place, ready
to be evaluated in another context, such as the data frame.

In practice, this is useful when you have a data-variable in an
env-variable (such as a function argument).\preformatted{# Force-defuse all function arguments that might contain
# data-variables by embracing them with {{ }}
mean_by <- function(data, by, var) {
  data \%>\%
    group_by({{ by }}) \%>\%
    summarise(avg = mean({{ var }}, na.rm = TRUE))
}

# The env-variables `by` and `var` are forced but defused.
# The data-variables they contain are evaluated by dplyr later on
# in data context.
iris \%>\% mean_by(by = Species, var = Sepal.Width)
}
}

Use \code{qq_show()} to experiment with forcing operators. \code{qq_show()}
defuses its input, processes all forcing operators, and prints the
result with \code{\link[=expr_print]{expr_print()}} to reveal objects inlined in the
expression by the forcing operators.
}
\section{Forcing names}{


When a function takes multiple named arguments
(e.g. \code{dplyr::mutate()}), it is difficult to supply a variable as
name. Since the LHS of \code{=} is \link[=nse-defuse]{defused}, giving the name
of a variable results in the argument having the name of the
variable rather than the name stored in that variable. This problem
of forcing evaluation of names is exactly what the \code{!!} operator is
for.

Unfortunately R is very strict about the kind of expressions
supported on the LHS of \code{=}. This is why rlang interprets the
walrus operator \code{:=} as an alias of \code{=}. You can use it to supply
names, e.g. \code{a := b} is equivalent to \code{a = b}. Since its syntax is
more flexible you can also force names on its LHS:\preformatted{name <- "Jane"

list2(!!name := 1 + 2)
exprs(!!name := 1 + 2)
quos(!!name := 1 + 2)
}

Like \code{=}, the \code{:=} operator expects strings or symbols on its LHS.

Currently, forcing names with \code{:=} only works in top level
expressions. These are all valid:\preformatted{exprs(!!nm := x)
tibble(!!nm := x)
list2(!!nm := x)
}

But deep-forcing names isn't supported:\preformatted{expr(this(is(deep(!!nm := x))))
exprs(this(is(deep(!!nm := x))))
}
}

\section{Theory}{


Formally, \code{quo()} and \code{expr()} are quasiquotation functions, \code{!!}
is the unquote operator, and \code{!!!} is the unquote-splice operator.
These terms have a rich history in Lisp languages, and live on in
modern languages like
\href{https://docs.julialang.org/en/v1/manual/metaprogramming/}{Julia}
and
\href{https://docs.racket-lang.org/reference/quasiquote.html}{Racket}.
}

\section{Life cycle}{

\itemize{
\item Calling \code{UQ()} and \code{UQS()} with the rlang namespace qualifier is
deprecated as of rlang 0.3.0. Just use the unqualified forms
instead:\preformatted{# Bad
rlang::expr(mean(rlang::UQ(var) * 100))

# Ok
rlang::expr(mean(UQ(var) * 100))

# Good
rlang::expr(mean(!!var * 100))
}

Supporting namespace qualifiers complicates the implementation of
unquotation and is misleading as to the nature of unquoting
operators (which are syntactic operators that operates at
quotation-time rather than function calls at evaluation-time).
\item \code{UQ()} and \code{UQS()} were soft-deprecated in rlang 0.2.0 in order
to make the syntax of quasiquotation more consistent. The prefix
forms are now \code{`!!`()} and \code{`!!!`()} which is
consistent with other R operators (e.g. \code{`+`(a, b)} is the
prefix form of \code{a + b}).

Note that the prefix forms are not as relevant as before because
\code{!!} now has the right operator precedence, i.e. the same as
unary \code{-} or \code{+}. It is thus safe to mingle it with other
operators, e.g. \code{!!a + !!b} does the right thing. In addition the
parser now strips one level of parentheses around unquoted
expressions. This way \code{(!!"foo")(...)} expands to \code{foo(...)}.
These changes make the prefix forms less useful.

Finally, the named functional forms \code{UQ()} and \code{UQS()} were
misleading because they suggested that existing knowledge about
functions is applicable to quasiquotation. This was reinforced by
the visible definitions of these functions exported by rlang and
by the tidy eval parser interpreting \code{rlang::UQ()} as \code{!!}. In
reality unquoting is \emph{not} a function call, it is a syntactic
operation. The operator form makes it clearer that unquoting is
special.
}
}

\examples{
# Interpolation with {{  }} is the easiest way to forward
# arguments to tidy eval functions:
if (is_attached("package:dplyr")) {

# Forward all arguments involving data frame columns by
# interpolating them within other data masked arguments.
# Here we interpolate `arg` in a `summarise()` call:
my_function <- function(data, arg) {
  summarise(data, avg = mean({{ arg }}, na.rm = TRUE))
}

my_function(mtcars, cyl)
my_function(mtcars, cyl * 10)

# The  operator is just a shortcut for `!!enquo()`:
my_function <- function(data, arg) {
  summarise(data, avg = mean(!!enquo(arg), na.rm = TRUE))
}

my_function(mtcars, cyl)

}

# Quasiquotation functions quote expressions like base::quote()
quote(how_many(this))
expr(how_many(this))
quo(how_many(this))

# In addition, they support unquoting. Let's store symbols
# (i.e. object names) in variables:
this <- sym("apples")
that <- sym("oranges")

# With unquotation you can insert the contents of these variables
# inside the quoted expression:
expr(how_many(!!this))
expr(how_many(!!that))

# You can also insert values:
expr(how_many(!!(1 + 2)))
quo(how_many(!!(1 + 2)))


# Note that when you unquote complex objects into an expression,
# the base R printer may be a bit misleading. For instance compare
# the output of `expr()` and `quo()` (which uses a custom printer)
# when we unquote an integer vector:
expr(how_many(!!(1:10)))
quo(how_many(!!(1:10)))

# This is why it's often useful to use qq_show() to examine the
# result of unquotation operators. It uses the same printer as
# quosures but does not return anything:
qq_show(how_many(!!(1:10)))


# Use `!!!` to add multiple arguments to a function. Its argument
# should evaluate to a list or vector:
args <- list(1:3, na.rm = TRUE)
quo(mean(!!!args))

# You can combine the two
var <- quote(xyz)
extra_args <- list(trim = 0.9, na.rm = TRUE)
quo(mean(!!var , !!!extra_args))


# The plural versions have support for the `:=` operator.
# Like `=`, `:=` creates named arguments:
quos(mouse1 := bernard, mouse2 = bianca)

# The `:=` is mainly useful to unquote names. Unlike `=` it
# supports `!!` on its LHS:
var <- "unquote me!"
quos(!!var := bernard, mouse2 = bianca)


# All these features apply to dots captured by enquos():
fn <- function(...) enquos(...)
fn(!!!args, !!var := penny)


# Unquoting is especially useful for building an expression by
# expanding around a variable part (the unquoted part):
quo1 <- quo(toupper(foo))
quo1

quo2 <- quo(paste(!!quo1, bar))
quo2

quo3 <- quo(list(!!quo2, !!!syms(letters[1:5])))
quo3
}
