% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convert_MR.R
\name{convert_MR}
\alias{convert_MR}
\title{Convert between units of absolute, mass-specific, or area-specific metabolic
rates}
\usage{
convert_MR(
  x,
  from = NULL,
  to = NULL,
  S = NULL,
  t = NULL,
  P = NULL,
  quiet = FALSE
)
}
\arguments{
\item{x}{numeric value or vector, or object of class \code{convert_rate} or
\code{convert_rate.ft}. Contains the rate(s) to be converted.}

\item{from}{string. The unit of the input metabolic rate(s). Should be in the
correct order: Oxygen/Time, Oxygen/Time/Mass or Oxygen/Time/Area. If \code{x} is
a \code{convert_rate} or \code{convert_rate.ft} object this is extracted
automatically and any input specified via \code{from} is ignored.}

\item{to}{string. The unit to convert the metabolic rate(s) to. Should be in
the correct order: Oxygen/Time, Oxygen/Time/Mass or Oxygen/Time/Area. See
\code{unit_args()}.}

\item{S}{numeric. Salinity (ppt). Defaults to NULL. Used in conversion of
some oxygen units. Freshwater should be entered as \code{S = 0}. If \code{x} is a
\code{convert_rate} or \code{convert_rate.ft} object this is extracted automatically
if present and any other input ignored.}

\item{t}{numeric. Temperature(°C). Defaults to NULL. Used in conversion of
some oxygen units. If \code{x} is a \code{convert_rate} or \code{convert_rate.ft} object
this is extracted automatically if present and any other input ignored.}

\item{P}{numeric. Pressure (bar). Used in conversion of some oxygen units.
Defaults to a standard value of 1.013253 bar. If \code{x} is a \code{convert_rate} or
\code{convert_rate.ft} object this is extracted automatically if present and any
other input ignored.}

\item{quiet}{logical. Suppresses the summary of the converted rates printed
to the console. Default is \code{FALSE}.}
}
\value{
If the \code{x} input rates are a numeric value or vector, output is a
numeric value or vector of the same length. If \code{x} is a \code{convert_rate} or
\code{convert_rate.ft} object, output is a new \code{convert_rate} or
\code{convert_rate.ft} object with all rates in the \verb{$summary} table,
\verb{$rate.output} and \verb{$output.unit} elements converted to the new units. This
allows you to convert results of analyses to a different unit without
having to repeat the entire analysis.
}
\description{
Converts metabolic rates to a different unit. These can be absolute rates
(i.e. whole chamber or whole specimen e.g. \code{mg/h}), mass-specific rates (i.e.
normalised by specimen mass  e.g. \code{mg/h/kg}), or area-specific rates (i.e.
normalised by specimen surface area e.g. \code{mg/h/cm2}). Input rates can be a
numeric value or vector, in which case the converted rates are output as a
numeric of the same length in the new units. Alternatively, input can be an
object of class \code{convert_rate()} or \code{convert_rate.ft()}, in which case a new
\code{convert_rate} or \code{convert_rate.ft} object is returned with all rates in the
\verb{$summary} and other elements converted to the new units. This allows you to
convert results of analyses to a different unit without having to repeat the
entire analysis.
}
\details{
\subsection{Units}{

Units are specified using \code{from} and \code{to}. These should be in the sequence
\emph{Oxygen-Time} (e.g. \code{"mg/h"}) for absolute rates, \emph{Oxygen-Time-Mass} (e.g.
\code{"mg/h/kg"}) for mass-specific rates, and \emph{Oxygen-Time-Area} (e.g.
\code{"mg/h/cm2"}) for surface area-specific rates. If \code{x} is a \code{convert_rate} or
\code{convert_rate.ft} object the \code{from} unit is extracted automatically.

A fuzzy string matching algorithm is used to accept various unit formatting
styles. For example, \code{"mg/h"}, \code{"mg/H"}, \code{"mg hr-1"}, \code{"milligram per hour"}
are all parsed the same. See \code{unit_args()} for details of accepted units and
their formatting.

Note some units require salinity (\code{S}) and temperature (\code{t}) to perform the
conversion. For freshwater experiments, salinity should be entered as zero
(i.e. \code{S = 0}). These conversions also require the atmospheric pressure
(\code{P}). If not entered the default value of 1.013253 bar (standard pressure at
sea level) is used. If \code{x} is a \code{convert_rate} or \code{convert_rate.ft} object,
\code{S}, \code{t}, and \code{P} are extracted automatically if they are present (they may
not be if the original rate conversion did not require them). They are also
saved to the \verb{$inputs} element of the output object.
}

\subsection{More details}{

For \code{convert_rate} or \code{convert_rate.ft} inputs the primary \verb{$rate.output}
element is converted to the new unit and the \verb{$output.unit} also updated to
this new unit. These columns are also updated in the \verb{$summary} table, and in
addition the \code{rate.abs} column and, if relevant, the \code{rate.m.spec} or
\code{rate.a.spec} column. Note, the \verb{$call} element is updated to the
\code{convert_MR} call and the original call to \code{convert_rate} or
\code{convert_rate.ft} replaced. The \verb{$inputs} element will still contain the
original inputs, with the \code{output.unit} updated to the new \code{to} unit. In
addition \code{S}, \code{t}, and \code{P} are added if they weren't already present.
}

\subsection{More}{

For additional help, documentation, vignettes, and more visit the \code{respR}
website at \url{https://januarharianto.github.io/respR/}
}
}
\examples{
# Convert a numeric absolute rate to a different unit
convert_MR(-0.09, from = 'mg/min', to = 'umol/hr')

# Convert a vector of absolute rates to a different unit
convert_MR(c(-0.090, -0.081, -0.098),
           from = 'mg/min', to = 'umol/hr')

# Convert to a unit which requires S, t, & P
convert_MR(-0.09, from = 'mg/min', to = 'ml/hour',
           S = 0, t = 20, P = 1.01)

# Convert mass-specific rates
convert_MR(-0.09, from = 'mg/min/g', to = 'ml/hour/kg',
           S = 0, t = 20, P = 1.01)

# Convert area-specific rates
convert_MR(-0.09, from = 'mg/min/mm2', to = 'ml/hour/cm2',
           S = 0, t = 20, P = 1.01)

# Convert from units largely only used in older papers.
# E.g. cubic cm (e.g. Tang 1933, Head 1962)
convert_MR(0.1, from = 'cc/hr/gm', to = 'mg/hr/g',
           S = 30, t = 20, P = 1.01)
convert_MR(0.6, from = 'cm3/hr', to = 'mg/hr',
           S = 28, t = 12, P = 1.01)
# uL (e.g. Zeuthen 1953, Newell & Northcroft 1967)
convert_MR(400, from = 'ul/hr', to = 'mg/hr',
           S = 30, t = 15, P = 1.01)
convert_MR(0.5, from = 'ul/hr/mg', to = 'mg/hr/g',
           S = 0, t = 20, P = 1.01)
# mm3 (e.g. Newell & Roy 1973)
convert_MR(1.5, from = 'mm3/hr', to = 'mg/hr',
           S = 30, t = 15, P = 1.01)

# Convert rates in a 'convert_rate' object. This avoids having to repeat
# an entire analysis to see the output in different units.

# Make a convert_rate object
cnv_rt.obj <- urchins.rd \%>\%
  auto_rate() \%>\%
  convert_rate(oxy.unit = "mg/L",
               time.unit = "min",
               output.unit = "mg/hr/kg",
               volume = 1,
               mass = 0.05)

# Now convert all results to "umol/min/g".
# The 'from' units are identified automatically from the object.
cnv_rt.obj.new <- convert_MR(cnv_rt.obj,
                             to = "umol/min/g")

# Compare the two:
summary(cnv_rt.obj)
summary(cnv_rt.obj.new)
}
