#' @name importEvents
#' @title Import Events to REDCap
#' 
#' @description This method allows you to import Events into a project or to 
#'   update existing Events' attributes, such as the event name, days offset, 
#'   etc. The unique event name of an Event cannot be changed because it is 
#'   auto-generated by REDCap. Please note that the only way to update an 
#'   existing Event is to provide the unique_event_name attribute, 
#'   and if the unique_event_name attribute is missing for an Event being 
#'   imported (when override=0), it will assume it to be a new Event that
#'   should be created. Notice: Because of the 'override' parameter's 
#'   destructive nature, this method may only use override=1 for projects 
#'   in Development status.
#'   
#' @param rcon a \code{redcapConnection} object. 
#' @param event_data \code{data.frame}. Must have columns "event_name"
#'   and "arm_num". To modify existing events, it must also have a column
#'   "unique_event_name". It may optionally have columns for 
#'   "day_offset", "offset_min", "offset_max".
#' @param override \code{logical(1)}. Defaults to \code{FALSE}, which will
#'   only add new events or modify existing events. When \code{TRUE}, all
#'   existing events are deleted and replaced with the events in 
#'   \code{event_data}.
#' @param refresh \code{logical(1)}. When \code{TRUE}, cached event data will 
#'   be refreshed after the import.
#' @param ... Additional arguments to pass to other methods. 
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}
#'   
#' @export

importEvents <- function(rcon, 
                         event_data, 
                         override = FALSE, 
                         ...){
  UseMethod("importEvents")
}

#' @rdname importEvents
#' @export

importEvents <- function(rcon, 
                         event_data, 
                         override       = FALSE, 
                         refresh        = TRUE,
                         ..., 
                         error_handling = getOption("redcap_error_handling"), 
                         config         = list(), 
                         api_param      = list()){
  ###################################################################
  # Argument Validation
  
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon, 
                          classes = "redcapApiConnection", 
                          add = coll)
  
  checkmate::assert_data_frame(x = event_data, 
                               add = coll)
  
  checkmate::assert_logical(x = override, 
                            len = 1, 
                            any.missing = FALSE, 
                            add = coll)
  
  checkmate::assert_logical(x = refresh, 
                            len = 1, 
                            add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling, 
                                        choices = c("null", "error"), 
                                        .var.name = "error_handling",
                                        add = coll)
  
  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
  
  checkmate::assert_subset(x = names(event_data), 
                           choices = names(REDCAP_EVENT_STRUCTURE), # Defined in redcapDataStructure.R 
                           add = coll)
  
  checkmate::reportAssertions(coll)
  
  ###################################################################
  # Make API Body List
  
  body <- list(content = "event", 
               action = "import", 
               override = as.numeric(override), 
               format = "csv", 
               returnFormat = "csv", 
               data = writeDataForImport(event_data))
  
  body <- body[lengths(body) > 0]
  
  ###################################################################
  # Call the API
  
  response <- makeApiCall(rcon, 
                          body = c(body, api_param), 
                          config = config)
  
  if (response$status_code != 200) return(redcap_error(response, error_handling))
  
  message(sprintf("Events imported: %s", 
                  as.character(response)))
  
  if (refresh && rcon$has_events()){
    rcon$refresh_events()
    rcon$refresh_projectInformation()
  }
}
