% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rddensity.R
\name{rddensity}
\alias{rddensity}
\title{Manipulation Testing Using Local Polynomial Density Estimation}
\usage{
rddensity(X, c = 0, p = 2, q = 0, kernel = "", fitselect = "",
  h = c(), bwselect = "", vce = "", all = FALSE)
}
\arguments{
\item{X}{Numeric vector or one dimensional matrix / data frame, the running variable.}

\item{c}{Numeric, specifies the threshold or cutoff value in the support of \code{X},
which determes the two samples (e.g., control and treatment units in RD settings).  Default
is \code{0}.}

\item{p}{Integer, specifies the order of the local-polynomial used to construct the density
point estimators.  Default is \code{2} (local quadratic approximation).}

\item{q}{Integer, specifies the order of the local-polynomial used to construct the
bias-corrected density point estimators.  Default is \code{p+1} (local cubic approximation).}

\item{kernel}{String, specifies the kernel function used to construct the local-polynomial
estimator(s). Options are: \code{"triangular"}, \code{"epanechnikov"}, and \code{"uniform"}. Default is
\code{"triangular"}.}

\item{fitselect}{String, specifies whether restrictions should be imposed. Options are:
\code{"unrestricted"} for density estimation without any restrictions (two-sample, unrestricted
inference). This is the default option. \code{"restricted"} for density estimation assuming
equal c.d.f. and higher-order derivatives.}

\item{h}{Numeric, specifies the bandwidth used to construct the density estimators on the
two sides of the cutoff. If not specified, the bandwidth is computed by the companion
command \code{\link{rdbwdensity}}. If two bandwidths are specified, the first bandwidth
is used for the data below the cutoff and the second bandwidth is used for the data above the cutoff.}

\item{bwselect}{String, specifies the bandwidth selection procedure to be used. Options are:
\code{"each"} bandwidth selection based on MSE of each density separately (two distinct bandwidths).
\code{"diff"} bandwidth selection based on MSE of difference of densities (one common bandwidth).
\code{"sum"} bandwidth selection based on MSE of sum of densities (one common bandwidth).
\code{"comb"} (this is the default option) bandwidth is selected as a combination of the alternatives above: for \code{fitselect = "unrestricted"},
it selects \code{median(each,diff,sum)}; for \code{fitselect = "restricted"}, it selects \code{min(diff,sum)}.}

\item{vce}{String, specifies the procedure used to compute the variance-covariance matrix estimator. Options are:
\code{"plugin"} for asymptotic plug-in standard errors. \code{"jackknife"} for jackknife standard errors. This
is the default option.}

\item{all}{Boolean, if specified, rddensity reports two testing procedures (given choices \code{fitselect}
and \code{bwselect}): Conventional test statistic (not valid when using MSE-optimal bandwidth choice).
Robust bias-corrected statistic.}
}
\value{
\item{hat}{\code{left}/\code{right}: density estimate to the left/right of cutoff; \code{diff}: difference in
  estimated densities on the two sides of cutoff.}
\item{sd_asy}{\code{left}/\code{right}: standard error for the estimated density to the left/right of the
  cutoff; \code{diff}: standard error for the difference in estimated densities. (Based on
  asymptotic formula.)}
\item{sd_jk}{\code{left}/\code{right}: standard error for the estimated density to the left/right of the
  cutoff; \code{diff}: standard error for the difference in estimated densities. (Based on the
  jackknife method.)}
\item{test}{\code{t_asy}/\code{t_jk}: t-statistic for the density discontinuity test, with standard error
  based on asymptotic formula or the jackknife; \code{p_asy}/\code{p_jk}: p-value for the density
  discontinuity test, with standard error based on asymptotic formula or the jackknife.}
\item{hat_p}{Same as \code{hat}, without bias correction (only available when \code{all=TRUE}).}
\item{sd_asy_p}{Same as \code{sd_asy}, without bias correction (only available when \code{all=TRUE}).}
\item{sd_jk_p}{Same as \code{sd_jk}, without bias correction (only available when \code{all=TRUE}).}
\item{test_p}{Same as \code{test}, without bias correction (only available when \code{all=TRUE}).}
\item{N}{\code{full}: full sample size; \code{left}/\code{right}: sample size to the left/right of the cutoff;
  \code{eff_left}/\code{eff_right}: effective sample size to the left/right of the cutoff (this depends
  on the bandwidth).}
\item{h}{\code{left}/\code{right}: bandwidth used to the left/right of the cutoff.}
\item{opt}{Collects the options used, includes: \code{fitselect}, \code{kernel}, \code{bwselectl},
  \code{bwselect}, \code{hscale}, \code{vce}, \code{c}, \code{p}, \code{q}, \code{all}.
  See options for \code{rddensity}.}
\item{X_min}{\code{left}/\code{right}: the samllest observation to the left/right of the cutoff.}
\item{X_max}{\code{left}/\code{right}: the largest observation to the left/right of the cutoff.}
}
\description{
\code{rddensity} implements manipulation testing procedures using the local
  polynomial density estimator proposed in Cattaneo, Jansson and Ma (2019).
  For a review on manipulation testing see McCrary (2008).

Companion command: \code{\link{rdbwdensity}} for data-driven bandwidth selection, and
  \code{\link{rdplotdensity}} for density plot.
  A companion \code{Stata} package is described in Cattaneo, Jansson and Ma (2018).

Related Stata and R packages useful for inference in regression discontinuity (RD)
  designs are described at \url{https://sites.google.com/site/rdpackages}.
}
\examples{
# Continuous Density
set.seed(42)
x <- rnorm(2000, mean = -0.5)
summary(rddensity(X = x, vce="jackknife"))

# Discontinuous density
x[x>0] <- x[x>0] * 2
summary(rddensity(X = x, vce="jackknife"))

}
\references{
M.D. Cattaneo, M. Jansson and X. Ma. (2018). \href{https://sites.google.com/site/rdpackages/rddensity/Cattaneo-Jansson-Ma_2018_Stata.pdf}{Manipulation Testing based on Density Discontinuity}.  \emph{Stata Journal} 18(1): 234-261.

M.D. Cattaneo, M. Jansson and X. Ma. (2019).  \href{https://arxiv.org/abs/1811.11512}{Simple Local Polynomial Density Estimators}. \emph{Journal of the American Statistical Association}, forthcoming.

J. McCrary. (2008). Manipulation of the Running Variable in the Regression Discontinuity Design: A Density Test. \emph{Journal of Econometrics} 142(2): 698-714.
}
\seealso{
\code{\link{rdbwdensity}}, \code{\link{rdplotdensity}}
}
\author{
Matias D. Cattaneo, Princeton University  \email{cattaneo@princeton.edu}.

Michael Jansson, University of California Berkeley.  \email{mjansson@econ.berkeley.edu}.

Xinwei Ma (maintainer), University of California San Diego. \email{x1ma@ucsd.edu}.
}
