#' @title Bland Altman Plots
#'
#' @description Construct and plot a Bland Altman plot in ggplot2.
#'
#' @details
#' Providing a \code{data.frame} with two columns, the function returns a ggplot
#' version of a Bland Altman plot with the specified confidence intervals.
#'
#' Two ways to call the plotting function.  If you submit a \code{data.frame}
#' \code{qblandaltman} then the data needed to produce the Bland Altman plot is
#' automatically generated by a call to \code{qblandaltman_build_data_frame}.
#' Alternatively, you may call \code{qblandaltman_build_data_frame} directly and
#' then call \code{qblandaltman}.  This might be helpful if you are putting
#' multiple Bland Altman plots together into one ggplot object.  See Examples.
#'
#' @param x a \code{data.frame} with two columns.  If a \code{data.frame}
#' with more than two columns is used only the first two columns will be used.
#' @param alpha (Defaults to 0.05) place (1 - alpha)*100% confidence levels to
#' place on the plot. 
#' @param generate_data logical, defaults to TRUE.  If TRUE, then the call to
#' \code{qblandaltman_build_data_frame} is done automatically for you.  If
#' FALSE, then you should explicitly call \code{qblandaltman_build_data_frame}
#' before calling \code{qblandaltman}.
#'
#' @return a ggplot.  Minimal aesthetics have been used so that the user may
#' modify the graphic as desired with ease.
#'
#' @references
#' Altman, Douglas G., and J. Martin Bland. "Measurement in medicine: the analysis
#' of method comparison studies." The statistician (1983): 307-317.
#' 
#' Bland, J. Martin, and DouglasG Altman. "Statistical methods for assessing
#' agreement between two methods of clinical measurement." The lancet 327, no. 8476
#' (1986): 307-310.
#'
#' @example examples/blandaltman.R
#' @export   
#' @rdname qblandaltman
qblandaltman <- function(x, alpha = getOption("qwraps2_alpha", 0.05), generate_data = TRUE) { 

  if (is.null(attr(x, "qwraps2_generated"))) { 
    if (generate_data) {
      x <- qblandaltman_build_data_frame(x, alpha)
    }
  }

  ggplot2::ggplot(x) + 
  ggplot2::aes_string(x = 'avg', y = 'diff') + 
  ggplot2::geom_point() + 
  ggplot2::geom_hline(ggplot2::aes_string(yintercept = 'lcl'), lty = 2) + 
  ggplot2::geom_hline(ggplot2::aes_string(yintercept = 'ucl'), lty = 2) + 
  ggplot2::geom_hline(ggplot2::aes_string(yintercept = 'mean_diff'), lty = 3) 
}

#' @export
#' @rdname qblandaltman
qblandaltman_build_data_frame <- function(x, alpha = getOption("qwraps2_alpha", 0.05)) { 
  rtn <-
    dplyr::mutate(data.frame(x1 = x[[1]], x2 = x[[2]]),
                  avg       = (.data$x1 + .data$x2) / 2, 
                  diff      = (.data$x2 - .data$x1), 
                  mean_diff = mean(.data$diff), 
                  sd_diff   = stats::sd(.data$diff), 
                  lcl       = .data$mean_diff + stats::qnorm(alpha / 2) * .data$sd_diff,
                  ucl       = .data$mean_diff + stats::qnorm(1 - alpha / 2) * .data$sd_diff) 
  rtn <- dplyr::tbl_df(rtn)

  attr(rtn, "qwraps2_generated") = TRUE

  return(rtn)
}

