% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nlf.R
\name{nlf}
\alias{nlf}
\alias{nlf_objfun}
\alias{nlf_objfun,missing-method}
\alias{nlf_objfun,ANY-method}
\alias{nlf_objfun,data.frame-method}
\alias{nlf_objfun,pomp-method}
\alias{nlf_objfun,nlf_objfun-method}
\title{Nonlinear forecasting}
\usage{
\S4method{nlf_objfun}{data.frame}(
  data,
  est = character(0),
  lags,
  nrbf = 4,
  ti,
  tf,
  seed = NULL,
  transform.data = identity,
  period = NA,
  tensor = TRUE,
  fail.value = NA_real_,
  params,
  rinit,
  rprocess,
  rmeasure,
  ...,
  verbose = getOption("verbose")
)

\S4method{nlf_objfun}{pomp}(
  data,
  est = character(0),
  lags,
  nrbf = 4,
  ti,
  tf,
  seed = NULL,
  transform.data = identity,
  period = NA,
  tensor = TRUE,
  fail.value = NA,
  ...,
  verbose = getOption("verbose")
)

\S4method{nlf_objfun}{nlf_objfun}(
  data,
  est,
  lags,
  nrbf,
  ti,
  tf,
  seed = NULL,
  period,
  tensor,
  transform.data,
  fail.value,
  ...,
  verbose = getOption("verbose", FALSE)
)
}
\arguments{
\item{data}{either a data frame holding the time series data,
or an object of class \sQuote{pomp},
i.e., the output of another \pkg{pomp} calculation.
Internally, \code{data} will be coerced to an array with storage-mode \code{double}.}

\item{est}{character vector; the names of parameters to be estimated.}

\item{lags}{A vector specifying the lags to use when constructing the nonlinear autoregressive prediction model.
The first lag is the prediction interval.}

\item{nrbf}{integer scalar;
the number of radial basis functions to be used at each lag.}

\item{ti, tf}{required numeric values.
NLF works by generating simulating long time series from the model.
The simulated time series will be from \code{ti} to \code{tf}, with the same sampling frequency as the data.
\code{ti} should be chosen large enough so that transient dynamics have died away.
\code{tf} should be chosen large enough so that sufficiently many data points are available to estimate the nonlinear forecasting model well.
An error will be generated unless the data-to-parameter ratio exceeds 10 and
a warning will be given if the ratio is smaller than 30.}

\item{seed}{integer.
When fitting, it is often best to fix the seed of the random-number generator (RNG).
This is accomplished by setting \code{seed} to an integer.
By default, \code{seed = NULL}, which does not alter the RNG state.}

\item{transform.data}{optional function.
If specified, forecasting is performed using data and model simulations transformed by this function.
By default, \code{transform.data} is the identity function,
i.e., no transformation is performed.
The main purpose of \code{transform.data} is to achieve approximately multivariate normal forecasting errors.
If the data are univariate, \code{transform.data} should take a scalar and return a scalar.
If the data are multivariate, \code{transform.data} should assume a vector input and return a vector of the same length.}

\item{period}{numeric; \code{period=NA} means the model is nonseasonal.
\code{period > 0} is the period of seasonal forcing.
\code{period <= 0} is equivalent to \code{period = NA}.}

\item{tensor}{logical;
if FALSE, the fitted model is a generalized additive model with time mod period as one of the predictors,
i.e., a gam with time-varying intercept.
If TRUE, the fitted model is a gam with lagged state variables as predictors and time-periodic coefficients, constructed using tensor products of basis functions of state variables with basis
functions of time.}

\item{fail.value}{optional numeric scalar;
if non-\code{NA}, this value is substituted for non-finite values of the objective function.
It should be a large number (i.e., bigger than any legitimate values the objective function is likely to take).}

\item{params}{optional; named numeric vector of parameters.
This will be coerced internally to storage mode \code{double}.}

\item{rinit}{simulator of the initial-state distribution.
This can be furnished either as a C snippet, an \R function, or the name of a pre-compiled native routine available in a dynamically loaded library.
Setting \code{rinit=NULL} sets the initial-state simulator to its default.
For more information, see \link[=rinit_spec]{rinit specification}.}

\item{rprocess}{simulator of the latent state process, specified using one of the \link[=rprocess_spec]{rprocess plugins}.
Setting \code{rprocess=NULL} removes the latent-state simulator.
For more information, see \link[=rprocess_spec]{rprocess specification for the documentation on these plugins}.}

\item{rmeasure}{simulator of the measurement model, specified either as a C snippet, an \R function, or the name of a pre-compiled native routine available in a dynamically loaded library.
Setting \code{rmeasure=NULL} removes the measurement model simulator.
For more information, see \link[=rmeasure_spec]{rmeasure specification}.}

\item{...}{additional arguments are passed to \code{\link{pomp}}.}

\item{verbose}{logical; if \code{TRUE}, diagnostic messages will be printed to the console.}
}
\value{
\code{nlf_objfun} constructs a stateful objective function for NLF estimation.
Specfically, \code{nlf_objfun} returns an object of class \sQuote{nlf_objfun}, which is a function suitable for use in an \code{\link[stats]{optim}}-like optimizer.
In particular, this function takes a single numeric-vector argument that is assumed to contain the parameters named in \code{est}, in that order.
When called, it will return the negative log quasilikelihood.
It is a stateful function:
Each time it is called, it will remember the values of the parameters and its estimate of the log quasilikelihood.
}
\description{
Parameter estimation by maximum simulated quasi-likelihood.
}
\details{
Nonlinear forecasting (NLF) is an \sQuote{indirect inference} method.
The NLF approximation to the log likelihood of the data series is computed by simulating data from a model, fitting a nonlinear autoregressive model to the simulated time series, and quantifying the ability of the resulting fitted model to predict the data time series.
The nonlinear autoregressive model is implemented as a generalized additive model (GAM), conditional on lagged values, for each observation variable.
The errors are assumed multivariate normal.

The NLF objective function constructed by \code{nlf_objfun} simulates long time series (\code{nasymp} is the number of observations in the simulated times series), perhaps after allowing for a transient period (\code{ntransient} steps).
It then fits the GAM for the chosen lags to the simulated time series.
Finally, it computes the quasi-likelihood of the data under the fitted GAM.

NLF assumes that the observation frequency (equivalently the time between successive observations) is uniform.
}
\section{Periodically-forced systems (seasonality)}{

Unlike other \pkg{pomp} estimation methods, NLF cannot accommodate general time-dependence in the model via explicit time-dependence or dependence on time-varying covariates.
However, NLF can accommodate periodic forcing.
It does this by including forcing phase as a predictor in the nonlinear autoregressive model.
To accomplish this, one sets \code{period} to the period of the forcing (a positive numerical value).
In this case, if \code{tensor = FALSE}, the effect is to add a periodic intercept in the autoregressive model.
If \code{tensor = TRUE}, by contrast, the fitted model includes time-periodic coefficients,
constructed using tensor products of basis functions of observables with
basis functions of time.
}

\section{Note for Windows users}{


Some Windows users report problems when using C snippets in parallel computations.
These appear to arise when the temporary files created during the C snippet compilation process are not handled properly by the operating system.
To circumvent this problem, use the \code{\link[=pomp]{cdir}} and \code{\link[=pomp]{cfile}} options to cause the C snippets to be written to a file of your choice, thus avoiding the use of temporary files altogether.
}

\section{Important Note}{

Since \pkg{pomp} cannot guarantee that the \emph{final} call an optimizer makes to the function is a call \emph{at} the optimum, it cannot guarantee that the parameters stored in the function are the optimal ones.
Therefore, it is a good idea to evaluate the function on the parameters returned by the optimization routine, which will ensure that these parameters are stored.
}

\section{Warning! Objective functions based on C snippets}{

If you use C snippets (see \code{\link{Csnippet}}), a dynamically loadable library will be built.
As a rule, \pkg{pomp} functions load this library as needed and unload it when it is no longer needed.
The stateful objective functions are an exception to this rule.
For efficiency, calls to the objective function do not execute \code{\link{pompLoad}} or \code{\link{pompUnload}}:
rather, it is assumed that \code{\link{pompLoad}} has been called before any call to the objective function.
When a stateful objective function using one or more C snippets is created, \code{\link{pompLoad}} is called internally to build and load the library:
therefore, within a single \R session, if one creates a stateful objective function, one can freely call that objective function and (more to the point) pass it to an optimizer that calls it freely, without needing to call \code{\link{pompLoad}}.
On the other hand, if one retrieves a stored objective function from a file, or passes one to another \R session, one must call \code{\link{pompLoad}} before using it.
\strong{Failure to do this will typically result in a segmentation fault (i.e., it will crash the \R session).}
}

\examples{
\donttest{

  if (require(subplex)) {

    ricker() |>
      nlf_objfun(est=c("r","sigma","N_0"),lags=c(4,6),
        partrans=parameter_trans(log=c("r","sigma","N_0")),
        paramnames=c("r","sigma","N_0"),
        ti=100,tf=2000,seed=426094906L) -> m1

    subplex(par=log(c(20,0.5,5)),fn=m1,control=list(reltol=1e-4)) -> out

    m1(out$par)
    coef(m1)
    plot(simulate(m1))

  }
}
}
\references{
\Ellner1998

\Kendall1999

\Kendall2005
}
\seealso{
\code{\link[stats]{optim}} \code{\link[subplex]{subplex}} \code{\link[nloptr]{nloptr}}

More on \pkg{pomp} estimation algorithms:
\code{\link{abc}()},
\code{\link{bsmc2}()},
\code{\link{estimation_algorithms}},
\code{\link{mif2}()},
\code{\link{pmcmc}()},
\code{\link{pomp-package}},
\code{\link{probe_match}},
\code{\link{spect_match}}

More on methods based on summary statistics: 
\code{\link{abc}()},
\code{\link{basic_probes}},
\code{\link{probe}()},
\code{\link{probe_match}},
\code{\link{spect}()},
\code{\link{spect_match}}

More on maximization-based estimation methods:
\code{\link{mif2}()},
\code{\link{probe_match}},
\code{\link{spect_match}},
\code{\link{traj_match}}
}
\author{
Stephen P. Ellner, Bruce E. Kendall, Aaron A. King
}
\concept{estimation methods}
\concept{methods based on maximization}
\concept{nonlinear forecasting}
\concept{summary statistic-based methods}
