\name{Utilities for reproducibility}
\alias{bake}
\alias{stew}
\alias{freeze}
\title{Tools for reproducible computations.}
\description{
  On cooking shows, recipes requiring lengthy baking or stewing are prepared beforehand.
  The \code{bake} and \code{stew} functions perform analogously:
  an \R computation is performed and stored in a named file.
  If the function is called again and the file is present, the computation is not executed; rather, the results are loaded from the file in which they were previously stored.
  Moreover, via their optional \code{seed} argument, \code{bake} and \code{stew} can control the pseudorandom-number generator (RNG) for greater reproducibility.
  After the computation is finished, these functions restore the pre-existing RNG state to avoid side effects.

  The \code{freeze} function doesn't save results, but does set the RNG state to the specified value and restore it after the computation is complete.
}
\usage{
bake(file, expr, seed, kind = NULL, normal.kind = NULL)
stew(file, expr, seed, kind = NULL, normal.kind = NULL)
freeze(expr, seed, kind = NULL, normal.kind = NULL)
}
\arguments{
  \item{file}{
    Name of the binary data file in which the result will be stored or retrieved, as appropriate.
    For \code{bake}, this will contain a single \R object and hence be an RDS file (extension \sQuote{rds});
    for \code{stew}, this will contain one or more named \R objects and hence be an RDA file (extension \sQuote{rda}).
  }
  \item{expr}{
    Expression to be evaluated.
  }
  \item{seed, kind, normal.kind}{
    Optional.
    To set the state and, optionally, kind of RNG used.
    See \code{\link{set.seed}}.
  }
}
\examples{
  \dontrun{
  bake(file="example1.rds",{
    x <- runif(1000)
    mean(x)
  })

  stew(file="example2.rda",{
    x <- runif(10)
    y <- rnorm(n=10,mean=3*x+5,sd=2)
  })

  plot(x,y)  
  }

  freeze(runif(3),seed=5886730)
  freeze(runif(3),seed=5886730)
}
\details{
  Both \code{bake} and \code{stew} first test to see whether \code{file} exists.
  If it does, \code{bake} reads it using \code{\link{readRDS}} and returns the resulting object.
  By contrast, \code{stew} loads the file using \code{\link{load}} and copies the objects it contains into the user's workspace (or the environment of the call to \code{stew}).

  If \code{file} does not exist, then both \code{bake} and \code{stew} evaluate the expression \code{expr};
  they differ in the results that they save.
  \code{bake} saves the value of the evaluated expression to \code{file} as a single \R object.
  The name of that object is not saved.
  By contrast, \code{stew} creates a local environment within which \code{expr}is evaluated;
  all objects in that environment are saved (by name) in \code{file}.
}
\value{
  \code{bake} returns the value of the evaluated expression \code{expr}.
  Other objects created in the evaluation of \code{expr} are discarded along with the temporary, local environment created for the evaluation.

  The latter behavior differs from that of \code{stew}, which returns the names of the objects created during the evaluation of \code{expr}.
  After \code{stew} completes, these objects exist in the parent environment (that from which \code{stew} was called).

  \code{freeze} returns the value of evaluated expression \code{expr}.
  However, \code{freeze} evaluates \code{expr} within the parent environment, so other objects created  in the evaluation of \code{expr} will therefore exist after \code{freeze} completes.
}
\author{Aaron A. King}
