% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/action_levels.R
\name{action_levels}
\alias{action_levels}
\alias{warn_on_fail}
\alias{stop_on_fail}
\title{Set action levels: failure thresholds and functions to invoke}
\usage{
action_levels(warn_at = NULL, stop_at = NULL, notify_at = NULL, fns = NULL)

warn_on_fail(warn_at = 1)

stop_on_fail(stop_at = 1)
}
\arguments{
\item{warn_at, stop_at, notify_at}{The threshold number or fraction of
test units that can provide a \emph{fail} result before entering the
\code{warn}, \code{stop}, or \code{notify} failure states. If this a decimal value between
\code{0} and \code{1} then it's a proportional failure threshold (e.g., \code{0.15}
indicates that if 15\% percent of the test units are found to \emph{fail},
then the designated failure state is entered). Absolute values starting
from \code{1} can be used instead, and this constitutes an absolute failure
threshold (e.g., \code{10} means that if 10 of the test units are found to
\emph{fail}, the failure state is entered).}

\item{fns}{A named list of functions that is to be paired with the
appropriate failure states. The syntax for this list involves using failure
state names from the set of \code{warn}, \code{stop}, and \code{notify}. The functions
corresponding to the failure states are provided as formulas (e.g.,
\code{list(warn = ~ warning("Too many failures."))}. A series of expressions for
each named state can be used by enclosing the set of statements with \code{{ }}.}
}
\description{
The \code{action_levels()} function works with the \code{actions} argument that is
present in the \code{\link[=create_agent]{create_agent()}} function and in every validation step
function (which also has an \code{actions} argument). With it, we can provide
threshold \emph{fail} levels for any combination of \code{warn}, \code{stop}, or \code{notify}
states.

We can react to any entrance of a state by supplying corresponding functions
to the \code{fns} argument. They will undergo evaluation at the time when the
matching state is entered. If provided to \code{\link[=create_agent]{create_agent()}} then the policies
will be applied to every validation step, acting as a default for the
validation as a whole.

Calls of \code{action_levels()} could also be applied directly to any validation
step and this will act as an override if set also in \code{\link[=create_agent]{create_agent()}}. Usage
of \code{action_levels()} is required to have any useful side effects (i.e.,
warnings, throwing errors) in the case of validation functions operating
directly on data (e.g., \code{mtcars \%>\% col_vals_lt("mpg", 35)}). There are two
helper functions that are convenient when using validation functions directly
on data (the \code{agent}-less workflow): \code{warn_on_fail()} and \code{stop_on_fail()}.
These helpers either warn or stop (default failure threshold for each is set
to \code{1}), and, they do so with informative warning or error messages. The
\code{stop_on_fail()} helper is applied by default when using validation functions
directly on data (more information on this is provided in \emph{Details}).
}
\details{
The output of the \code{action_levels()} call in \code{actions} will be interpreted
slightly differently if using an \emph{agent} or using validation functions
directly on a data table. For convenience, when working directly on data, any
values supplied to \code{warn_at} or \code{stop_at} will be automatically given a stock
\code{warning()} or \code{stop()} function. For example using
\code{small_table \%>\% col_is_integer("date")} will provide a detailed stop message
by default, indicating the reason for the failure. If you were to supply the
\code{fns} for \code{stop} or \code{warn} manually then the stock functions would be
overridden. Furthermore, if \code{actions} is NULL in this workflow (the default),
\strong{pointblank} will use a \code{stop_at} value of \code{1} (providing a detailed,
context-specific error message if there are any \emph{fail} units). We can
absolutely suppress this automatic stopping behavior by at each validation
step by setting \code{active = FALSE}. In this interactive data case, there is no
stock function given for \code{notify_at}. The \code{notify} failure state is less
commonly used in this workflow as it is in the \emph{agent}-based one.

When using an \emph{agent}, we often opt to not use any functions in \code{fns} as the
\code{warn}, \code{stop}, and \code{notify} failure states will be reported on when using
\code{create_agent_report()} (and, usually that's sufficient). Instead, using the
\code{end_fns} argument is a better choice since that scheme provides useful data
on the entire interrogation, allowing for finer control on side effects and
reducing potential for duplicating any side effects.
}
\section{Function ID}{

1-5
}

\examples{

# For these examples, we will use the
# included `small_table` dataset
small_table

# Create an `action_levels` object
# with fractional values for the
# `warn`, `stop`, and `notify` states
al <- 
  action_levels(
    warn_at = 0.2,
    stop_at = 0.8,
    notify_at = 0.5
  )
  
# A summary of settings for the `al`
# object is shown by printing it
al

# Create a pointblank agent and
# apply the `al` object to `actions`;
# add two validation steps and
# interrogate the `small_table`
agent_1 <-
  create_agent(
    tbl = small_table,
    actions = al
  ) \%>\%
  col_vals_gt(
    vars(a), value = 2
  ) \%>\%
  col_vals_lt(
    vars(d), value = 20000
  ) \%>\%
  interrogate()

# The report from the agent will show
# that the `warn` state has been entered
# for the first validation step but not
# the second one; we can confirm this
# in the console by inspecting the
# `warn` component in the agent's x-list
x_list <- get_agent_x_list(agent_1)
x_list$warn

# Applying the `action_levels` object
# to the agent means that all validation
# steps will inherit these settings but
# we can override this by applying
# another such object to the validation
# step instead (this time using the
# `warn_on_fail()` shorthand)
agent_2 <-
  create_agent(
    tbl = small_table,
    actions = al
  ) \%>\%
  col_vals_gt(
    vars(a), value = 2,
    actions = warn_on_fail(warn_at = 0.5)
  ) \%>\%
  col_vals_lt(
    vars(d), value = 20000
  ) \%>\%
  interrogate()

# In this case, the first validation
# step has a less stringent failure
# threshold for the `warn` state and it's
# high enough that the condition is not
# entered; this can be confirmed in the
# console through inspection of the
# x-list `warn` component
x_list <- get_agent_x_list(agent_2)
x_list$warn

if (interactive()) {

# In the context of using validation
# functions directly on data (i.e., no
# involvement of an agent) we want to
# trigger warnings and raise errors; the
# following will yield a warning if
# it is executed (returning the
# `small_table` data)
small_table \%>\%
  col_vals_gt(
    vars(a), value = 2,
    actions = warn_on_fail(warn_at = 2)
  )

# With the same pipeline, not supplying
# anything for `actions` (it's `NULL` by
# default) will have the same effect as
# using `stop_on_fail(stop_at = 1)` 
small_table \%>\%
  col_vals_gt(vars(a), value = 2)

small_table \%>\%
  col_vals_gt(
    vars(a), value = 2,
    actions = stop_on_fail(stop_at = 1)
  )

# This is because the `stop_on_fail()`
# call is auto-injected in the default
# case (when operating on data) for your
# convenience; behind the scenes a
# 'secret agent' uses 'covert actions':
# all so you can type less

}

}
\seealso{
Other Planning and Prep: 
\code{\link{create_agent}()},
\code{\link{create_informant}()},
\code{\link{db_tbl}()},
\code{\link{draft_validation}()},
\code{\link{file_tbl}()},
\code{\link{scan_data}()},
\code{\link{tbl_get}()},
\code{\link{tbl_source}()},
\code{\link{tbl_store}()},
\code{\link{validate_rmd}()}
}
\concept{Planning and Prep}
