# ----------------------
# Author: Andreas Alfons
#         KU Leuven
# ----------------------

#' Plot resampling-based prediction error results
#' 
#' Plot results of resampling-based prediction error measures.
#' 
#' For objects with multiple columns of prediction error results, conditional 
#' plots are produced.
#' 
#' @param x  an object inheriting from class \code{"perry"} or 
#' \code{"perrySelect"} that contains prediction error results, or a data frame 
#' containing all necessary information for plotting (as generated by the 
#' corresponding \code{\link[=fortify.perry]{fortify}} method).
#' @param object  an object inheriting from class \code{"perry"} or 
#' \code{"perrySelect"}.
#' @param method  a character string specifying the type of plot.  Possible 
#' values are \code{"box"} to create a box plot, \code{"density"} to create a 
#' smooth density plot, \code{"dot"} to create a dot plot, or \code{"line"} to 
#' plot the (average) results for each model as a connected line (for objects 
#' inheriting from class \code{"perrySelect"}).  Note that the first two plots 
#' are only meaningful in case of repeated resampling.  The default is to use 
#' \code{"box"} in case of repeated resampling and \code{"dot"} otherwise.  In 
#' any case, partial string matching allows supply abbreviations of the 
#' accepted values.
#' @param subset  a character, integer or logical vector indicating the subset 
#' of models for which to plot the prediction error results.
#' @param select  a character, integer or logical vector indicating the columns 
#' of prediction error results to be plotted.
#' @param seFactor  a numeric value giving the multiplication factor of the 
#' standard error for displaying error bars in dot plots or line plots.  Error 
#' bars in those plots can be suppressed by setting this to \code{NA}.
#' @param mapping  an aesthetic mapping to override the default behavior (see 
#' \code{\link{aes}} or \code{\link{aes_string}})
#' @param facets  a faceting formula to override the default behavior.  If 
#' supplied, \code{\link[ggplot2]{facet_wrap}} or 
#' \code{\link[ggplot2]{facet_grid}} is called depending on whether the formula 
#' is one-sided or two-sided.
#' @param \dots  for the generic function \code{perryPlot}, additional 
#' arguments to be passed down to methods.  For the \code{"perry"} and 
#' \code{"perrySelect"} methods of \code{perryPlot}, additional arguments 
#' to be passed down to the default method.  For the default method of 
#' \code{perryPlot}, additional arguments to be passed down to 
#' \code{\link[ggplot2]{geom_boxplot}}, \code{\link[ggplot2]{geom_density}}, 
#' \code{\link[ggplot2]{geom_pointrange}} or 
#' \code{\link[ggplot2]{geom_line}}.  For the methods of \code{plot}, 
#' additional arguments to be passed down to \code{perryPlot}.
#' 
#' @return  
#' An object of class \code{"ggplot"} (see \code{\link[ggplot2]{ggplot}}).
#' 
#' @note Duplicate indices in \code{subset} or \code{select} are removed such 
#' that all models and prediction error results are unique.
#' 
#' @author Andreas Alfons
#' 
#' @seealso \code{\link{perryFit}}, \code{\link{perrySelect}}, 
#' \code{\link{perryTuning}}, \code{\link[ggplot2]{ggplot}}, 
#' \code{\link[ggplot2]{autoplot}}, \code{\link[graphics]{plot}}
#' 
#' @example inst/doc/examples/example-perryPlot.R
#' 
#' @keywords hplot
#' 
#' @import ggplot2
#' @export

perryPlot <- function(x, ...) UseMethod("perryPlot")


#' @rdname perryPlot
#' @method perryPlot perry
#' @export

perryPlot.perry <- function(x, method = c("box", "density", "dot"), 
        select = NULL, seFactor = NA, ...) {
    # initializations
    if(x$splits$R == 1) {
        choices <- eval(formals()[["method"]])
        if(identical(method, choices)) method <- "dot"
        else method <- match.arg(method, "dot")
    } else method <- match.arg(method)
    # extract data for plotting
    reps <- method %in% c("box", "density")
    data <- fortify(x, select=select, reps=reps, seFactor=seFactor)
    # call default method
    perryPlot(data, method=method, ...)
}


#' @rdname perryPlot
#' @method perryPlot perrySelect
#' @export

perryPlot.perrySelect <- function(x, 
        method = c("box", "density", "dot", "line"), subset = NULL, 
        select = NULL, seFactor = x$seFactor, ...) {
    # initializations
    if(x$splits$R == 1) {
        choices <- eval(formals()[["method"]])
        if(identical(method, choices)) method <- "dot"
        else method <- match.arg(method, c("dot", "line"))
    } else method <- match.arg(method)
    # extract data for plotting
    reps <- method %in% c("box", "density")
    data <- fortify(x, subset=subset, select=select, 
        reps=reps, seFactor=seFactor)
    # call default method
    perryPlot(data, method=method, ...)
}


#' @rdname perryPlot
#' @method perryPlot default
#' @export

perryPlot.default <- function(x, method = c("box", "density", "dot", "line"), 
        mapping, facets = attr(x, "facets"), ...) {
    # initializations
    if(is.null(x$Lower) && is.null(x$Upper)) method <- match.arg(method)
    else {
        choices <- eval(formals()[["method"]])
        if(identical(method, choices)) method <- "dot"
        else method <- match.arg(method, c("dot", "line"))
    }
    # call function for selected plot
    if(method == "box") boxPlot(x, mapping=mapping, facets=facets, ...)
    else if(method == "density") 
        densityPlot(x, mapping=mapping, facets=facets, ...)
    else if(method == "line") linePlot(x, mapping=mapping, facets=facets, ...)
    else dotPlot(x, mapping=mapping, facets=facets, ...)
}


#' @rdname perryPlot
#' @method autoplot perry
#' @export

autoplot.perry <- function(object, ...) perryPlot(object, ...)


#' @rdname perryPlot
#' @method autoplot perrySelect
#' @export

autoplot.perrySelect <- function(object, ...) perryPlot(object, ...)


#' @rdname perryPlot
#' @method plot perry
#' @export

plot.perry <- function(x, ...) perryPlot(x, ...)


#' @rdname perryPlot
#' @method plot perrySelect
#' @export

plot.perrySelect <- function(x, ...) perryPlot(x, ...)

# ----------

boxPlot <- function(data, mapping, facets = attr(data, "facets"), 
        main = NULL, xlab = NULL, ylab = NULL, ...) {
    # define aesthetic mapping for box plot
    if(missing(mapping)) mapping <- aes_string(x="Fit", y="PE", group="Fit")
    # define default axis label
    if(is.null(ylab)) ylab <- "Prediction error"
    # generate plot
    p <- ggplot(data, mapping) + geom_boxplot(...) + 
        opts(title=main) + labs(x=xlab, y=ylab)
    if(!is.null(facets)) {
        # split plot into different panels
        if(length(facets) == 2) p <- p + facet_wrap(facets) 
        else p <- p + facet_grid(facets)
    }
    p
}

# ----------

densityPlot <- function(data, mapping, facets = attr(data, "facets"), 
        main = NULL, xlab = NULL, ylab = NULL, ...) {
    # define aesthetic mapping for box plot
    if(missing(mapping)) {
        if(nlevels(data[, "Fit"]) > 1 || length(unique(data[, "Fit"])) > 1) 
            mapping <- aes_string(x="PE", group="Fit", color="Fit")
        else mapping <- aes_string(x="PE")
    }
    # define default axis label
    if(is.null(xlab)) xlab <- "Prediction error"
    # generate plot
    p <- ggplot(data, mapping) + geom_density(...) + 
        opts(title=main) + labs(x=xlab, y=ylab)
    if(is.numeric(data[, "Fit"])) 
        p <- p + scale_color_continuous(breaks=unique(data[, "Fit"]))
    if(!is.null(facets)) {
        # split plot into different panels
        if(length(facets) == 2) p <- p + facet_wrap(facets) 
        else p <- p + facet_grid(facets)
    }
    p
}

# ----------

dotPlot <- function(data, mapping, facets = attr(data, "facets"), 
        main = NULL, xlab = NULL, ylab = NULL, ...) {
    # define aesthetic mapping for box plot
    if(missing(mapping)) 
        mapping <- aes_string(x="Fit", y="PE", ymin="Lower", ymax="Upper")
    # define default axis label
    if(is.null(ylab)) ylab <- "Prediction error"
    # generate plot
    p <- ggplot(data, mapping) + geom_pointrange(...) + 
        opts(title=main) + labs(x=xlab, y=ylab)
    if(!is.null(facets)) {
        # split plot into different panels
        if(length(facets) == 2) p <- p + facet_wrap(facets) 
        else p <- p + facet_grid(facets)
    }
    p
}

# ----------

linePlot <- function(data, mapping, facets = attr(data, "facets"), 
        main = NULL, xlab = NULL, ylab = NULL, ...) {
    # define aesthetic mapping for box plot
    if(missing(mapping)) 
        mapping <- aes_string(x="Fit", y="PE", ymin="Lower", ymax="Upper")
    # define default axis label
    if(is.null(ylab)) ylab <- "Prediction error"
    # generate plot
    p <- ggplot(data, mapping) + geom_line(...) + geom_pointrange(...) + 
        opts(title=main) + labs(x=xlab, y=ylab)
    if(!is.null(facets)) {
        # split plot into different panels
        if(length(facets) == 2) p <- p + facet_wrap(facets) 
        else p <- p + facet_grid(facets)
    }
    p
}
