% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/regmest_regression.R
\name{regmest_cv}
\alias{regmest_cv}
\alias{adamest_cv}
\title{Cross-validation for (Adaptive) Elastic Net M-Estimates}
\usage{
regmest_cv(
  x,
  y,
  standardize = TRUE,
  lambda,
  cv_k,
  cv_repl = 1,
  cv_metric = c("tau_size", "mape", "rmspe", "auroc"),
  fit_all = TRUE,
  cl = NULL,
  ...
)

adamest_cv(x, y, alpha, alpha_preliminary = 0, exponent = 1, ...)
}
\arguments{
\item{x}{\code{n} by \code{p} matrix of numeric predictors.}

\item{y}{vector of response values of length \code{n}.
For binary classification, \code{y} should be a factor with 2 levels.}

\item{standardize}{whether to standardize the \code{x} variables prior to fitting the PENSE estimates.
Can also be set to \code{"cv_only"}, in which case the input data is not standardized, but the
training data in the CV folds is scaled to match the scaling of the input data.
Coefficients are always returned on the original scale. This can fail for variables with a large
proportion of a single value (e.g., zero-inflated data). In this case, either compute with
\code{standardize = FALSE} or standardize the data manually.}

\item{lambda}{optional user-supplied sequence of penalization levels. If given and not \code{NULL}, \code{nlambda} and
\code{lambda_min_ratio} are ignored.}

\item{cv_k}{number of folds per cross-validation.}

\item{cv_repl}{number of cross-validation replications.}

\item{cv_metric}{either a string specifying the performance metric to use, or a function to evaluate prediction
errors in a single CV replication. If a function, the number of arguments define the data the function receives.
If the function takes a single argument, it is called with a single numeric vector of prediction errors.
If the function takes two or more arguments, it is called with the predicted values as first argument
and the true values as second argument.
The function must always return a single numeric value quantifying the prediction performance.
The order of the given values corresponds to the order in the input data.}

\item{fit_all}{If \code{TRUE}, fit the model for all penalization levels. Otherwise, only at penalization
level with smallest average CV performance.}

\item{cl}{a \link[parallel:makeCluster]{parallel} cluster. Can only be used if \code{ncores = 1}, because multi-threading
can not be used in parallel R sessions on the same host.}

\item{...}{
  Arguments passed on to \code{\link[=regmest]{regmest}}
  \describe{
    \item{\code{scale}}{fixed scale of the residuals.}
    \item{\code{nlambda}}{number of penalization levels.}
    \item{\code{lambda_min_ratio}}{Smallest value of the penalization level as a fraction of the largest level (i.e., the
smallest value for which all coefficients are zero). The default depends on the sample
size relative to the number of variables and \code{alpha}. If more observations than variables
are available, the default is \code{1e-3 * alpha}, otherwise \code{1e-2 * alpha}.}
    \item{\code{penalty_loadings}}{a vector of positive penalty loadings (a.k.a. weights) for different penalization of each
coefficient. Only allowed for \code{alpha} > 0.}
    \item{\code{starting_points}}{a list of staring points, created by \code{\link[=starting_point]{starting_point()}}. The starting points are shared
among all penalization levels.}
    \item{\code{intercept}}{include an intercept in the model.}
    \item{\code{add_zero_based}}{also consider the 0-based regularization path in addition to the given starting points.}
    \item{\code{cc}}{cutoff constant for Tukey's bisquare \eqn{\rho} function.}
    \item{\code{eps}}{numerical tolerance.}
    \item{\code{explore_solutions}}{number of solutions to compute up to the desired precision \code{eps}.}
    \item{\code{explore_tol}}{numerical tolerance for exploring possible solutions. Should be (much) looser than \code{eps} to
be useful.}
    \item{\code{max_solutions}}{only retain up to \code{max_solutions} unique solutions per penalization level.}
    \item{\code{comparison_tol}}{numeric tolerance to determine if two solutions are equal. The comparison is first done
on the absolute difference in the value of the objective function at the solution
If this is less than \code{comparison_tol}, two solutions are deemed equal if the squared difference
of the intercepts is less than \code{comparison_tol} and the squared \eqn{L_2} norm of the
difference vector is less than \code{comparison_tol}.}
    \item{\code{sparse}}{use sparse coefficient vectors.}
    \item{\code{ncores}}{number of CPU cores to use in parallel. By default, only one CPU core is used. May not be supported
on your platform, in which case a warning is given.}
    \item{\code{algorithm_opts}}{options for the MM algorithm to compute estimates. See \code{\link[=mm_algorithm_options]{mm_algorithm_options()}} for details.}
    \item{\code{mscale_bdp}}{options for the M-scale estimate used to standardize the predictors
(if \code{standardize = TRUE}).}
    \item{\code{mscale_opts}}{options for the M-scale estimate used to standardize the predictors
(if \code{standardize = TRUE}).}
  }}

\item{alpha}{elastic net penalty mixing parameter with \eqn{0 \le \alpha \le 1}. \code{alpha = 1} is the LASSO penalty,
and \code{alpha = 0} the Ridge penalty.}

\item{alpha_preliminary}{\code{alpha} parameter for the preliminary estimate.}

\item{exponent}{the exponent for computing the penalty loadings based on the preliminary estimate.}
}
\value{
a list with components:
\describe{
\item{\code{lambda}}{the sequence of penalization levels.}
\item{\code{scale}}{the used scale of the residuals.}
\item{\code{cvres}}{data frame of average cross-validated performance.}
\item{\code{cv_replications}}{matrix of cross-validated performance metrics, one column per replication.
Rows are in the same order as in \code{cvres}.}
\item{\code{call}}{the original call.}
\item{\code{estimates}}{the estimates fitted on the full data. Same format as returned by \code{\link[=regmest]{regmest()}}.}
}

the object returned by \code{adamest_cv()} has additional components
\describe{
\item{\code{preliminary}}{the CV results for the preliminary estimate.}
\item{\code{penalty_loadings}}{the penalty loadings used for the adaptive elastic net M-estimate.}
}
}
\description{
Perform (repeated) K-fold cross-validation for \code{\link[=regmest]{regmest()}}.

\code{adamest_cv()} is a convenience wrapper to compute adaptive elastic-net M-estimates.
}
\details{
The built-in CV metrics are
\describe{
\item{\code{"tau_size"}}{\eqn{\tau}-size of the prediction error, computed by \code{\link[=tau_size]{tau_size()}} (default).}
\item{\code{"mape"}}{Median absolute prediction error.}
\item{\code{"rmspe"}}{Root mean squared prediction error.}
\item{\code{"auroc"}}{Area under the receiver operator characteristic curve (actually 1 - AUROC). Only sensible for
binary responses.}
}

\code{adamest_cv()} is a convenience wrapper which performs 3 steps:
\enumerate{
\item compute preliminary estimates via \code{regmest_cv(..., alpha = alpha_preliminary)},
\item computes the penalty loadings from the estimate \code{beta} with best prediction performance by
\code{adamest_loadings = 1 / abs(beta)^exponent}, and
\item compute the adaptive PENSE estimates via \code{regmest_cv(..., penalty_loadings = adamest_loadings)}.
}
}
\examples{
# Compute the adaptive PENSE regularization path for Freeny's
# revenue data (see ?freeny)
data(freeny)
x <- as.matrix(freeny[ , 2:5])

## Either use the convenience function directly ...
ada_convenience <- adapense_cv(x, freeny$y, alpha = 0.5,
                               cv_repl = 2, cv_k = 4)

## ... or compute the steps manually:
# Step 1: Compute preliminary estimates with CV
preliminary_estimate <- pense_cv(x, freeny$y, alpha = 0,
                                 cv_repl = 2, cv_k = 4)
plot(preliminary_estimate, se_mult = 1)

# Step 2: Use the coefficients with best prediction performance
# to define the penality loadings:
prelim_coefs <- coef(preliminary_estimate, lambda = 'min')
pen_loadings <- 1 / abs(prelim_coefs[-1])

# Step 3: Compute the adaptive PENSE estimates and estimate
# their prediction performance.
ada_manual <- pense_cv(x, freeny$y, alpha = 0.5, cv_repl = 2,
                       cv_k = 4, penalty_loadings = pen_loadings)

# Visualize the prediction performance and coefficient path of
# the adaptive PENSE estimates (manual vs. automatic)
def.par <- par(no.readonly = TRUE)
layout(matrix(1:4, ncol = 2, byrow = TRUE))
plot(ada_convenience$preliminary)
plot(preliminary_estimate)
plot(ada_convenience)
plot(ada_manual)
par(def.par)
}
\seealso{
\code{\link[=regmest]{regmest()}} for computing regularized S-estimates without cross-validation.

\code{\link[=coef.pense_cvfit]{coef.pense_cvfit()}} for extracting coefficient estimates.

\code{\link[=plot.pense_cvfit]{plot.pense_cvfit()}} for plotting the CV performance or the regularization path.

Other functions to compute robust estimates with CV: 
\code{\link{pense_cv}()},
\code{\link{pensem_cv}()}

Other functions to compute robust estimates with CV: 
\code{\link{pense_cv}()},
\code{\link{pensem_cv}()}
}
\concept{functions to compute robust estimates with CV}
