#' Pedigree subgroups
#'
#' A collection of utility functions for identifying pedigree members with
#' certain properties.
#'
#' @param x A [ped()] object.
#' @param id,ids A character (or coercible to such) with one or several ID
#'   labels.
#' @param inclusive A logical indicating whether an individual should be counted
#'   among his or her own ancestors/descendants
#' @param internal A logical indicating whether `id` (or `ids`) refers to the
#'   internal order.
#' @param degree,removal Non-negative integers.
#' @param half a logical or NA. If TRUE (resp FALSE), only half (resp. full)
#'   siblings/cousins/nephews/nieces are returned. If NA, both categories are
#'   included.
#'
#' @return The functions `ancestors(x, id)` and `descendants(x, id)` return a
#'   vector containing the IDs of all ancestors (resp. descendants) of the
#'   individual `id` within the pedigree `x`. If `inclusive = TRUE`, `id` is
#'   included in the output.
#'
#'   For `commonAncestors(x, ids)` and `commonDescendants(x, ids)`, a vector
#'   containing the IDs of common ancestors to all of `ids`.
#'
#'   The functions `founders`, `nonfounders`, `males`, `females`, `leaves` each
#'   return a vector containing the IDs of all pedigree members with the wanted
#'   property. (Recall that a founder is a member without parents in the
#'   pedigree, and that a leaf is a member without children in the pedigree.)
#'
#'   The functions `father`, `mother`, `cousins`, `grandparents`,
#'   `nephews_nieces`, `children`, `parents`, `siblings`, `spouses`,
#'   `unrelated`, each returns a vector containing the IDs of all pedigree
#'   members having the specified relationship with `id`.
#'
#' @author Magnus Dehli Vigeland
#'
#' @examples
#'
#' x = ped(id = 2:9,
#'          fid = c(0,0,2,0,4,4,0,2),
#'          mid = c(0,0,3,0,5,5,0,8),
#'          sex = c(1,2,1,2,1,2,2,2))
#' stopifnot(setequal(spouses(x, 2), c(3,8)),
#'           setequal(children(x, 2), c(4,9)),
#'           setequal(descendants(x, 2), c(4,6,7,9)),
#'           setequal(leaves(x), c(6,7,9)))
#'
#' @name ped_subgroups
NULL


#' @rdname ped_subgroups
#' @export
founders = function(x, internal = FALSE) {
  is_fou = x$FIDX == 0
  if (internal) which(is_fou) else labels.ped(x)[is_fou]
}

#' @rdname ped_subgroups
#' @export
nonfounders = function(x, internal = FALSE) {
  is_nonfou = x$FIDX > 0
  if (internal) which(is_nonfou) else labels.ped(x)[is_nonfou]
}

#' @rdname ped_subgroups
#' @export
leaves = function(x, internal = FALSE) {
  if(is.singleton(x))
    leaves_int = 1L
  else
    leaves_int = (1:pedsize(x))[-c(x$FIDX, x$MIDX)]
  if (internal) leaves_int else labels.ped(x)[leaves_int]
}

#' @rdname ped_subgroups
#' @export
males = function(x, internal = FALSE) {
  m = x$SEX == 1
  if (internal) which(m) else labels.ped(x)[m]
}

#' @rdname ped_subgroups
#' @export
females = function(x, internal = FALSE) {
  f = x$SEX == 2
  if (internal) which(f) else labels.ped(x)[f]
}

#' @rdname ped_subgroups
#' @export
typedMembers = function(x, internal = FALSE) {
  if(is.pedList(x))
    return(unlist(lapply(x, typedMembers)))

  if (nMarkers(x) == 0)
    return(if(internal) integer(0) else character(0))

  allelematrix = do.call(cbind, x$MARKERS)
  emptyrows = rowSums(allelematrix != 0) == 0
  if(internal) which(!emptyrows) else labels.ped(x)[!emptyrows]
}

#' @rdname ped_subgroups
#' @export
untypedMembers = function(x, internal = FALSE) {
  if(is.pedList(x))
    return(unlist(lapply(x, untypedMembers)))

  if (nMarkers(x) == 0)
    return(if(internal) seq_len(pedsize(x)) else labels.ped(x))
  allelematrix = do.call(cbind, x$MARKERS)
  emptyrows = rowSums(allelematrix != 0) == 0
  if(internal) which(emptyrows) else labels.ped(x)[emptyrows]
}

#' @rdname ped_subgroups
#' @export
father = function(x, id, internal = FALSE) {
  if (!internal) id = internalID(x, id)
  fa = x$FIDX[id]
  if (internal) fa else labels.ped(x)[fa]
}

#' @rdname ped_subgroups
#' @export
mother = function(x, id, internal = FALSE) {
  if (!internal) id = internalID(x, id)
  mo = x$MIDX[id]
  if (internal) mo else labels.ped(x)[mo]
}

#' @rdname ped_subgroups
#' @export
children = function(x, id, internal = FALSE) {
    if (!internal) id = internalID(x, id)
    offs_int = (x$FIDX == id | x$MIDX == id)

    if (internal) which(offs_int) else labels.ped(x)[offs_int]
}

#' @rdname ped_subgroups
#' @export
offspring = children

#' @rdname ped_subgroups
#' @export
spouses = function(x, id, internal = FALSE) {
  # Returns a vector containing all individuals sharing offspring with <id>.
  if (!internal)  id = internalID(x, id)
  spous = switch(x$SEX[id] + 1,
                c(x$MIDX[x$FIDX == id], x$FIDX[x$MIDX == id]), # sex = 0
                x$MIDX[x$FIDX == id],                        # sex = 1
                x$FIDX[x$MIDX == id])                        # sex = 2
  spous_uniq = unique.default(spous)
  if (internal) spous_uniq else labels.ped(x)[spous_uniq]
}


#' @rdname ped_subgroups
#' @export
unrelated = function(x, id, internal = FALSE) {
  if (!internal)  id = internalID(x, id)
  ancs = c(id, ancestors(x, id))
    rel = unique.default(unlist(lapply(ancs, function(a) c(a, descendants(x, a, internal = FALSE)))))
    unrel = setdiff(labels.ped(x), rel)
    if (internal) internalID(x, unrel) else unrel
}


#' @rdname ped_subgroups
#' @export
parents = function(x, id, internal = FALSE) {
  if (!internal) id = internalID(x, id)
  parents_int = c(x$FIDX[id], x$MIDX[id])
  if (internal) parents_int else labels.ped(x)[parents_int]
}

#' @rdname ped_subgroups
#' @export
grandparents = function(x, id, degree = 2, internal = FALSE) {
  if (!internal)  id = internalID(x, id)

  nextgen = id
  for (i in seq_len(degree)) nextgen = c(x$FIDX[nextgen], x$MIDX[nextgen])
  if (internal) nextgen else labels.ped(x)[nextgen]
}

#' @rdname ped_subgroups
#' @export
siblings = function(x, id, half = NA, internal = FALSE) {
  if (!internal)  id = internalID(x, id)
  fa = x$FIDX[id]
  mo = x$MIDX[id]
  if (fa == 0 && mo == 0)
    return(if(internal) integer(0) else character(0))

  samefather = x$FIDX == fa
  samemother = x$MIDX == mo
  sib_int =
    if (isTRUE(half)) samefather | samemother
    else if (isFALSE(half)) xor(samefather, samemother)
    else if(is.na(half)) samefather & samemother
  sib_int[id] = FALSE
  if (internal) which(sib_int) else labels.ped(x)[sib_int]
}

#' @rdname ped_subgroups
#' @export
cousins = function(x, id, degree = 1, removal = 0, half = NA, internal = FALSE) {
  if (!internal)  id = internalID(x, id)
  gp = grandparents(x, id, degree = degree, internal = TRUE)
  gp = gp[gp > 0]
  if(length(gp) == 0)
    return(if(internal) integer(0) else character(0))

  uncles = unique.default(unlist(lapply(gp, function(a)
    siblings(x, a, half = half, internal = TRUE))))

  cous = uncles
  for (i in seq_len(degree + removal))
    cous = unique.default(unlist(lapply(cous, children, x = x, internal = TRUE)))

  if (internal) cous else labels.ped(x)[cous]
}

#' @rdname ped_subgroups
#' @export
nephews_nieces = function(x, id, removal = 1, half = NA, internal = FALSE) {
    cousins(x, id, degree = 0, removal = removal, half = half, internal = internal)
}

#' @rdname ped_subgroups
#' @export
ancestors = function(x, id, inclusive = FALSE, internal = FALSE) {
  # climbs upwards storing parents iteratively. (Not documented: Accepts id of length > 1)
  if (!internal)  id = internalID(x, id)

  FIDX = x$FIDX
  MIDX = x$MIDX
  ancest = if(inclusive) id else integer(0)

  up1 = c(FIDX[id], MIDX[id])
  up1 = up1[up1 > 0]
  while (length(up1)) {
    ancest = c(ancest, up1)
    up1 = c(FIDX[up1], MIDX[up1])
    up1 = up1[up1 > 0]
  }
  ancest = .mysortInt(unique.default(ancest))
  if (internal) ancest else labels.ped(x)[ancest]
}

#' @rdname ped_subgroups
#' @export
commonAncestors = function(x, ids, inclusive = FALSE, internal = FALSE) {
  if(length(ids) < 2)
    stop2("Argument `ids` must have length at least 2")

  anc = ancestors(x, ids[1], inclusive = inclusive, internal = internal)
  for(id in ids[-1]) {
    if(length(anc) == 0)
      break
    newanc = ancestors(x, id, inclusive = inclusive, internal = internal)
    anc = .myintersect(anc, newanc)
  }

  anc
}

#' @rdname ped_subgroups
#' @export
descendants = function(x, id, inclusive = FALSE, internal = FALSE) {
  if (!internal)  id = internalID(x, id)

  FIDX = x$FIDX
  MIDX = x$MIDX
  desc = if(inclusive) id else integer()

  nextoffs = id
  while(length(nextoffs)) {
      nextoffs = which(FIDX %in% nextoffs | MIDX %in% nextoffs)
      desc = c(desc, nextoffs)
  }
  desc = .mysortInt(unique.default(desc))
  if (internal) desc else labels.ped(x)[desc]
}

#' @rdname ped_subgroups
#' @export
commonDescendants = function(x, ids, inclusive = FALSE, internal = FALSE) {
  if(length(ids) < 2)
    stop2("Argument `ids` must have length at least 2")

  desc = descendants(x, ids[1], inclusive = inclusive, internal = internal)
  for(id in ids[-1]) {
    if(length(desc) == 0)
      break
    newdesc = descendants(x, id, inclusive = inclusive, internal = internal)
    desc = .myintersect(desc, newdesc)
  }

  desc
}

