\name{evaluation}
\alias{evaluation}
\alias{C_bd}
\alias{C_linzeta}

\title{
Multidimensional evaluation on posets
}
\description{
Given a partial order (arguments \code{profiles} and/or
\code{zeta}) and a selected \code{threshold},
the function returns an object of S3 class \code{parsec}, comprising the identification function and
different severity measures, computed by uniform sampling of the linear extensions of the poset, through a C implementation of the Bubley - Dyer (1999) algorithm.
}
\usage{
evaluation(
    profiles = NULL,
    threshold,
    error = 10^(-3),
    zeta = getzeta(profiles),
    weights = {
        if (!is.null(profiles)) 
            profiles$freq
        else rep(1, nrow(zeta))
    },
    distances = {
        n <- nrow(zeta)
        matrix(1, n, n) - diag(1, n)
    },
    linext = lingen(zeta),
    nit = floor({
        n <- nrow(zeta)
        n^5 * log(n) + n^4 * log(error^(-1))
    }),
    maxint = 2^31 - 1%,
    %inequality = FALSE
)
}

\arguments{
  \item{profiles}{
an object of S3 class \code{wprof}.
}
  \item{threshold}{
a vector identifying the threshold. It can be a vector of indexes (numeric),
a vector of profile names (character) or a boolean vector of length equal to the number of
profiles.
}
  \item{error}{
the "distance" from uniformity in the sampling distribution of linear extensions.
}
  \item{zeta}{
the incidence matrix of the poset. An object of S3 class \code{incidence}.
By default, extracted from \code{profiles}.
}
  \item{weights}{
weights assigned to profiles. If the argument \code{profiles} is not \code{NULL},
 weights are by default set equal to profile frequencies, otherwise they are
set equal to 1.
}
  \item{distances}{
matrix of distances between pairs of profiles. The matrix must be square, with dimensions
equal to the number of profiles. Even if the poset is
complete, the distance between two profiles is computed only if one
profile covers the other.
% Observe that the user can define two different distances in case of the
% profile A covers the profile B and in case case of the profile B covers the
% profile A.
}
  \item{linext}{
the linear extension initializing the sampling algorithm. By default, it is generated by \code{lingen(zeta)}. Alternatively, it can be provided by 
the user through a vector of profile positions.
}
  \item{nit}{
Number of ITerations in the Bubley-Dyer algorithm, by default evaluated from a formula of Karzanov and Khachiyan
based on the number of profiles and the argument \code{error} (see Bubley and Dyer, 1999).
}
  \item{maxint}{
Maximum integer. By default the maximum integer obtainable in a 32bit system.
This argument is used to group iterations and run the compiled
C code more times, so as to avoid memory indexing problems. Users can
set a lower value to \code{maxint} in case of low RAM availability.
}
  %\item{inequality}{
%boolean parameter (by default \code{FALSE}) to make the evaluation function return also a measure of inequality (which can make computations quite lengthy).
%}
}

\value{
\item{profiles}{an object of S3 class \code{wprof} reporting poset profiles
and their associated frequencies (number of statistical units in each profile).}

\item{number_of_profiles}{number of profiles.}

\item{number_of_variables}{number of variables.}

\item{incidence}{S3 class \code{incidence}, incidence matrix of the poset.}

\item{cover}{S3 class \code{cover}, cover matrix of the poset.}

\item{threshold}{boolean vector specifying whether a profile belongs to the threshold.}

\item{number_of_iterations}{number of iterations performed by the Bubley-Dyer algorithm.}

\item{rank_dist}{matrix reporting by rows the relative
frequency distributions of the ranks of each profile, over the set of sampled linear extensions.}

\item{thr_dist}{vector reporting the relative frequency a profile is used as threshold in the sampled linear extensions.}

\item{prof_w}{vector of weights assigned to each profile.}

\item{edg_w}{matrix of distances between profiles, used to evaluate the
gap measures.}

\item{idn_f}{vector reporting the identification function, computed as the fraction of sampled linear extensions where a profile is in the downset of the threshold.}

\item{svr_abs}{vector reporting, for each profile, the average graph distance
from the first profile above all threshold elements, over the sampled linear extensions.
In each linear extension, the distance is set equal to 0 for profiles above the threshold.}

\item{svr_rel}{equal to svr_abs divided by its
maximum, that is svr_abs of the minimal element in the linear extension.}

\item{wea_abs}{vector reporting, for each profile, the average graph distance
from the maximum threshold element, over the sampled linear extensions.
In each linear extension, the distance is set equal to 0 for profiles in the downset of threshold elements.}

\item{wea_rel}{the previous absolute distance is divided by its
maximum possible value, that is the absolute distance of the threshold
from the maximal element in the linear extension.}

\item{poverty_gap}{Population mean of svr_rel}

\item{wealth_gap}{Population mean of wea_rel}

%\item{inequality}{when the argument \code{inequality} is \code{TRUE}, the average value of the inequality index over the linear extensions (see Fattore and Arcagni, 2013).}
}
\references{
Bubley R., Dyer M. (1999), Faster random generation of linear extensions, Discrete Math., 201, 81-88.

Fattore M., Arcagni A. (2013), \href{http://meetings.sis-statistica.org/index.php/sis2013/ALV/paper/view/2633}{Measuring multidimensional polarization with ordinal data}, SIS 2013 Statistical Conference, BES-M3.1 - The BES and the challenges of constructing composite indicators dealing with equity and sustainability
}
\examples{
profiles <- var2prof(varlen = c(3, 2, 2))
threshold <- c("311", "112")

res <- evaluation(profiles, threshold, maxint = 10^5)

summary(res)
plot(res)
}