#' @title Confidence Intervals (CI)
#' @name ci.merMod
#'
#' @description Compute confidence intervals (CI) for frequentist models.
#'
#' @param x A statistical model.
#' @param ci Confidence Interval (CI) level. Default to 0.95 (95\%).
#' @param method For mixed models, can be \code{\link[=p_value_wald]{"wald"}} (default), \code{\link[=p_value_ml1]{"ml1"}} or \code{\link[=p_value_betwithin]{"betwithin"}}. For linear mixed model, can also be \code{\link[=p_value_satterthwaite]{"satterthwaite"}}, \code{\link[=p_value_kenward]{"kenward"}} or \code{"boot"} and \code{lme4::confint.merMod}). For (generalized) linear models, can be \code{"robust"} to compute confidence intervals based on robust covariance matrix estimation, and for generalized linear models, may also be \code{"profile"} (default) or \code{"wald"}.
#' @param ... Arguments passed down to \code{standard_error_robust()} when confidence intervals or p-values based on robust standard errors should be computed.
#' @inheritParams simulate_model
#' @inheritParams standard_error
#' @inheritParams p_value
#'
#' @return A data frame containing the CI bounds.
#'
#' @note \code{ci_robust()} resp. \code{ci(method = "robust")}
#'   rely on the \pkg{sandwich} or \pkg{clubSandwich} package (the latter if
#'   \code{vcov_estimation = "CR"} for cluster-robust standard errors) and will
#'   thus only work for those models supported by those packages.
#'
#' @examples
#' \donttest{
#' library(parameters)
#' if (require("glmmTMB")) {
#'   model <- glmmTMB(
#'     count ~ spp + mined + (1 | site),
#'     ziformula = ~mined,
#'     family = poisson(),
#'     data = Salamanders
#'   )
#'
#'   ci(model)
#'   ci(model, component = "zi")
#' }
#' }
#' @export
ci.merMod <- function(x,
                      ci = 0.95,
                      method = c("wald", "ml1", "betwithin", "satterthwaite", "kenward", "boot"),
                      ...) {
  method <- tolower(method)
  method <- match.arg(method)

  # Wald approx
  if (method == "wald") {
    out <- ci_wald(model = x, ci = ci, dof = Inf)

    # ml1 approx
  } else if (method == "ml1") {
    out <- ci_ml1(x, ci)

    # betwithin approx
  } else if (method == "betwithin") {
    out <- ci_betwithin(x, ci)

    # Satterthwaite
  } else if (method == "satterthwaite") {
    out <- ci_satterthwaite(x, ci)

    # Kenward approx
  } else if (method %in% c("kenward", "kr")) {
    out <- ci_kenward(x, ci)

    # bootstrapping
  } else if (method == "boot") {
    out <- lapply(ci, function(ci, x) .ci_boot_merMod(x, ci, ...), x = x)
    out <- do.call(rbind, out)
    row.names(out) <- NULL
  }

  out
}


#' @importFrom bayestestR ci
#' @export
bayestestR::ci


# Default Wald CI method ------------------------------------------------------


#' @rdname ci.merMod
#' @export
ci.default <- function(x, ci = .95, method = NULL, ...) {
  if (!is.null(method)) {
    method <- tolower(method)
  } else {
    method <- "wald"
  }

  if (method == "robust") {
    ci_wald(model = x, ci = ci, dof = Inf, robust = TRUE)
  } else if (method == "ml1") {
    ci_ml1(model = x, ci = ci)
  } else if (method == "betwithin") {
    ci_betwithin(model = x, ci = ci)
  } else {
    ci_wald(model = x, ci = ci, dof = Inf, robust = FALSE)
  }
}


#' @export
ci.mlm <- function(x, ci = .95, ...) {
  if (is.null(insight::find_weights(x))) {
    out <- lapply(ci, function(i) {
      .ci <- stats::confint(x, level = i, ...)
      rn <- rownames(.ci)
      .data_frame(
        Parameter = gsub("^(.*):(.*)", "\\2", rn),
        CI = i,
        CI_low = .ci[, 1],
        CI_high = .ci[, 2],
        Response = gsub("^(.*):(.*)", "\\1", rn)
      )
    })
    out <- .remove_backticks_from_parameter_names(do.call(rbind, out))
  } else {
    out <- .data_frame(ci_wald(x, ci = ci, ...), Response = insight::get_parameters(x)$Response)
  }

  out
}


#' @export
ci.bayesx <- function(x, ci = .95, ...) {
  ci_wald(model = x, ci = ci, dof = Inf, robust = FALSE, component = "conditional")
}


#' @export
ci.merModList <- function(x, ci = .95, ...) {
  ci_wald(model = x, ci = ci, dof = NULL, robust = FALSE, component = "conditional")
}


#' @export
ci.averaging <- function(x, ci = .95, component = c("conditional", "full"), ...) {
  component <- match.arg(component)
  ci_wald(model = x, ci = ci, dof = Inf, component = component)
}


#' @method ci lm
#' @export
ci.lm <- function(x, ci = .95, method = NULL, ...) {
  robust <- !is.null(method) && method == "robust"
  ci_wald(model = x, ci = ci, robust = robust, ...)
}

#' @export
ci.lm_robust <- ci.lm

#' @export
ci.comlmrob <- ci.lm

#' @export
ci.rq <- ci.lm

#' @export
ci.rqss <- ci.lm

#' @export
ci.crq <- ci.lm

#' @export
ci.nlrq <- ci.lm

#' @export
ci.BBmm <- ci.lm

#' @export
ci.BBreg <- ci.lm


#' @export
ci.gam <- function(x, ci = .95, ...) {
  ci_wald(model = x, ci = ci, ...)
}

#' @export
ci.scam <- ci.gam

#' @export
ci.mipo <- ci.gam

#' @export
ci.mira <- function(x, ci = .95, ...) {
  if (!requireNamespace("mice", quietly = TRUE)) {
    stop("Package 'mice' needed for this function to work. Please install it.")
  }
  ci(mice::pool(x), ci = ci, ...)
}

#' @export
ci.list <- function(x, ci = .95, ...) {
  if ("gam" %in% names(x)) {
    x <- x$gam
    class(x) <- c("gam", "lm", "glm")
    ci(x, ci = ci, ...)
  } else {
    return(NULL)
  }
}





# glm CI method with profiling -----------------------------------------------


#' @rdname ci.merMod
#' @method ci glm
#' @export
ci.glm <- function(x, ci = .95, method = c("profile", "wald", "robust"), ...) {
  method <- match.arg(method)
  if (method == "profile") {
    out <- lapply(ci, function(i) .ci_profiled(model = x, ci = i))
    out <- do.call(rbind, out)
  } else if (method == "robust") {
    out <- ci_wald(model = x, ci = ci, robust = TRUE, ...)
  } else {
    out <- ci_wald(model = x, ci = ci)
  }

  row.names(out) <- NULL
  out
}

#' @export
ci.negbin <- ci.glm

#' @export
ci.logistf <- ci.glm

#' @export
ci.mle2 <- ci.glm

#' @export
ci.mle <- ci.glm



#' @rdname ci.merMod
#' @export
ci.polr <- function(x, ci = .95, method = c("profile", "wald", "robust"), ...) {
  method <- match.arg(method)
  if (method == "profile") {
    out <- lapply(ci, function(i) .ci_profiled2(model = x, ci = i))
    out <- do.call(rbind, out)
  } else if (method == "robust") {
    out <- ci_wald(model = x, ci = ci, robust = TRUE, ...)
  } else {
    out <- ci_wald(model = x, ci = ci)
  }

  # for polr, profiled CI do not return CI for response levels
  # thus, we also calculate Wald CI and add missing rows to result

  out_missing <- ci_wald(model = x, ci = ci)
  missing_rows <- out_missing$Parameter %in% setdiff(out_missing$Parameter, out$Parameter)
  out <- rbind(out, out_missing[missing_rows, ])

  # fix names, to match standard error and p_value

  out$Parameter <- gsub("Intercept: ", "", out$Parameter, fixed = TRUE)
  row.names(out) <- NULL

  out
}






# Default Wald CI method with Inf dof -----------------------------------------


#' @export
ci.gamlss <- function(x, ci = .95, method = NULL, ...) {
  robust <- !is.null(method) && method == "robust"
  ci_wald(model = x, ci = ci, dof = Inf, robust = robust, ...)
}

#' @export
ci.speedglm <- ci.gamlss

#' @export
ci.cpglm <- ci.gamlss

#' @export
ci.cpglmm <- ci.gamlss

#' @export
ci.glmx <- ci.gamlss

#' @export
ci.glmmadmb <- ci.gamlss

#' @export
ci.fixest <- ci.gamlss

#' @export
ci.feglm <- ci.gamlss

#' @export
ci.speedlm <- ci.gamlss

#' @export
ci.glmrob <- ci.gamlss

#' @export
ci.plm <- ci.gamlss

#' @export
ci.LORgee <- ci.gamlss

#' @export
ci.truncreg <- ci.gamlss

#' @export
ci.ivreg <- ci.gamlss

#' @export
ci.gee <- ci.gamlss

#' @export
ci.tobit <- ci.gamlss

#' @export
ci.geeglm <- ci.gamlss

#' @export
ci.coxph <- ci.gamlss

#' @export
ci.aareg <- ci.gamlss

#' @export
ci.clm <- ci.gamlss

#' @export
ci.crch <- ci.gamlss

#' @export
ci.feis <- ci.gamlss

#' @export
ci.censReg <- ci.gamlss

#' @export
ci.survreg <- ci.gamlss

#' @export
ci.flexsurvreg <- ci.gamlss

#' @export
ci.coxme <- ci.gamlss

#' @export
ci.svyglm.nb <- ci.gamlss

#' @export
ci.lrm <- ci.gamlss

#' @export
ci.psm <- ci.gamlss

#' @export
ci.ols <- ci.gamlss

#' @export
ci.rms <- ci.gamlss

#' @export
ci.svyglm.zip <- ci.gamlss

#' @export
ci.vglm <- ci.gamlss

#' @export
ci.svyglm.glimML <- ci.gamlss

#' @export
ci.sem <- ci.gamlss


#' @rdname ci.merMod
#' @export
ci.mixor <- function(x, ci = .95, effects = c("all", "fixed", "random"), ...) {
  effects <- match.arg(effects)
  ci_wald(model = x, ci = ci, dof = Inf, effects = effects, robust = FALSE, ...)
}


#' @export
ci.gamm <- function(x, ci = .95, ...) {
  x <- x$gam
  class(x) <- c("gam", "lm", "glm")
  ci(x, ci = ci, ...)
}

#' @export
ci.gamm4 <- ci.gamm


#' @export
ci.multinom <- function(x, ci = .95, method = NULL, ...) {
  robust <- !is.null(method) && method == "robust"
  params <- insight::get_parameters(x)

  out <- ci_wald(model = x, ci = ci, dof = Inf, robust = robust, ...)

  if ("Response" %in% colnames(params)) {
    out$Response <- params$Response
  }

  out
}

#' @export
ci.brmultinom <- ci.multinom

#' @export
ci.bracl <- ci.multinom


#' @rdname ci.merMod
#' @export
ci.DirichletRegModel <- function(x, ci = .95, component = c("all", "conditional", "precision"), ...) {
  component <- match.arg(component)
  params <- insight::get_parameters(x, component = component)
  out <- ci_wald(model = x, ci = ci, dof = Inf, ...)

  if (is.null(out$Component)) {
    component <- "all"
  }
  if ("Response" %in% colnames(params)) {
    out$Response <- params$Response
  }
  if (component != "all") {
    out <- out[out$Component == component, ]
  }

  out
}




#' @rdname ci.merMod
#' @export
ci.HLfit <- function(x,
                     ci = 0.95,
                     method = c("wald", "ml1", "betwithin", "profile", "boot"),
                     iterations = 100,
                     ...) {
  method <- tolower(method)
  method <- match.arg(method)

  # Wald approx
  if (method == "wald") {
    out <- ci_wald(model = x, ci = ci, dof = Inf)

    # ml1 approx
  } else if (method == "ml1") {
    out <- ci_ml1(x, ci)

    # betwithin approx
  } else if (method == "betwithin") {
    out <- ci_betwithin(x, ci)

    # profiled
  } else if (method == "profile") {
    nparms <- n_parameters(x)
    conf <- stats::confint(x, parm = 1:nparms, level = ci, verbose = FALSE, boot_args = NULL)
    if (nparms == 1) {
      out <- as.data.frame(t(conf$interval))
    } else {
      out <- as.data.frame(do.call(rbind, lapply(conf, function(i) i$interval)))
    }
    colnames(out) <- c("CI_low", "CI_high")
    out$Parameter <- insight::find_parameters(x, effects = "fixed", flatten = TRUE)
    out$CI <- ci * 100
    out <- out[c("Parameter", "CI", "CI_low", "CI_high")]
  }

  #   # bootstrapping
  # } else if (method == "boot") {
  #   out <- stats::confint(x, parm = n_parameters(x), level = ci, verbose = FALSE, boot_args = list(nsim = iterations, showpbar = FALSE))
  # }

  out
}








# Zero-Inflated and Mixed models -----------------------------------------


#' @rdname ci.merMod
#' @export
ci.glmmTMB <- function(x,
                       ci = .95,
                       component = c("all", "conditional", "zi", "zero_inflated", "dispersion"),
                       method = c("wald", "ml1", "betwithin", "robust"),
                       verbose = TRUE,
                       ...) {
  method <- tolower(method)
  method <- match.arg(method)
  component <- match.arg(component)

  if (is.null(.check_component(x, component, verbose = verbose))) {
    return(NULL)
  }

  if (method == "robust") {
    ci_wald(model = x, ci = ci, dof = Inf, component = component, robust = TRUE)
  } else if (method == "wald") {
    ci_wald(model = x, ci = ci, dof = Inf, component = component, robust = FALSE)
  } else if (method == "ml1") {
    ci_ml1(model = x, ci = ci)
  } else if (method == "betwithin") {
    ci_betwithin(model = x, ci = ci)
  }
}

#' @rdname ci.merMod
#' @export
ci.zeroinfl <- ci.glmmTMB

#' @rdname ci.merMod
#' @export
ci.hurdle <- ci.glmmTMB

#' @export
ci.zerocount <- ci.glmmTMB


#' @rdname ci.merMod
#' @export
ci.MixMod <- function(x,
                      ci = .95,
                      component = c("all", "conditional", "zi", "zero_inflated"),
                      verbose = TRUE,
                      ...) {
  component <- match.arg(component)
  if (is.null(.check_component(x, component, verbose = verbose))) {
    return(NULL)
  }
  ci_wald(model = x, ci = ci, dof = Inf, component = component)
}


#' @export
ci.glmm <- function(x, ci = .95, effects = c("all", "fixed", "random"), ...) {
  effects <- match.arg(effects)
  ci_wald(model = x, ci = ci, dof = Inf, effects = effects, robust = FALSE)
}







# mfx models -----------------------------------------


#' @export
ci.logitor <- function(x, ci = .95, method = NULL, ...) {
  robust <- !is.null(method) && method == "robust"
  ci_wald(model = x$fit, ci = ci, robust = robust, ...)
}

#' @export
ci.poissonirr <- ci.logitor

#' @export
ci.negbinirr <- ci.logitor

#' @rdname ci.merMod
#' @export
ci.poissonmfx <- function(x, ci = .95, component = c("all", "conditional", "marginal"), method = NULL, ...) {
  component <- match.arg(component)
  robust <- !is.null(method) && method == "robust"
  ci_wald(model = x, ci = ci, component = component, robust = robust, ...)
}

#' @export
ci.negbinmfx <- ci.poissonmfx

#' @export
ci.logitmfx <- ci.poissonmfx

#' @export
ci.probitmfx <- ci.poissonmfx

#' @export
ci.betaor <- function(x, ci = .95, component = c("all", "conditional", "precision"), ...) {
  component <- match.arg(component)
  ci_wald(model = x$fit, ci = ci, dof = Inf, component = component)
}

#' @rdname ci.merMod
#' @export
ci.betamfx <- function(x, ci = .95, component = c("all", "conditional", "precision", "marginal"), method = NULL, ...) {
  component <- match.arg(component)
  robust <- !is.null(method) && method == "robust"
  ci_wald(model = x, ci = ci, component = component, robust = robust, ...)
}






# Special models -----------------------------------------


# ci.emm_list
# ci.emmGrid
# - implamented in bayestestR




#' @importFrom insight get_parameters model_info
#' @importFrom stats quantile
#' @export
ci.mediate <- function(x, ci = .95, ...) {
  info <- insight::model_info(x$model.y)
  alpha <- (1 + ci) / 2
  if (info$is_linear && !x$INT) {
    out <- data.frame(
      Parameter = c("ACME", "ADE", "Total Effect", "Prop. Mediated"),
      CI = 100 * ci,
      CI_low = c(
        stats::quantile(x$d0.sims, probs = 1 - alpha, names = FALSE),
        stats::quantile(x$z0.sims, probs = 1 - alpha, names = FALSE),
        stats::quantile(x$tau.sims, probs = 1 - alpha, names = FALSE),
        stats::quantile(x$n0.sims, probs = 1 - alpha, names = FALSE)
      ),
      CI_high = c(
        stats::quantile(x$d0.sims, probs = alpha, names = FALSE),
        stats::quantile(x$z0.sims, probs = alpha, names = FALSE),
        stats::quantile(x$tau.sims, probs = alpha, names = FALSE),
        stats::quantile(x$n0.sims, probs = alpha, names = FALSE)
      ),
      stringsAsFactors = FALSE
    )
  } else {
    out <- data.frame(
      Parameter = c(
        "ACME (control)", "ACME (treated)", "ADE (control)",
        "ADE (treated)", "Total Effect", "Prop. Mediated (control)",
        "Prop. Mediated (treated)", "ACME (average)", "ADE (average)",
        "Prop. Mediated (average)"
      ),
      CI = 100 * ci,
      CI_low = c(
        stats::quantile(x$d0.sims, probs = 1 - alpha, names = FALSE),
        stats::quantile(x$d1.sims, probs = 1 - alpha, names = FALSE),
        stats::quantile(x$z0.sims, probs = 1 - alpha, names = FALSE),
        stats::quantile(x$z1.sims, probs = 1 - alpha, names = FALSE),
        stats::quantile(x$tau.sims, probs = 1 - alpha, names = FALSE),
        stats::quantile(x$n0.sims, probs = 1 - alpha, names = FALSE),
        stats::quantile(x$n1.sims, probs = 1 - alpha, names = FALSE),
        stats::quantile(x$d.avg.sims, probs = 1 - alpha, names = FALSE),
        stats::quantile(x$z.avg.sims, probs = 1 - alpha, names = FALSE),
        stats::quantile(x$n.avg.sims, probs = 1 - alpha, names = FALSE)
      ),
      CI_high = c(
        stats::quantile(x$d0.sims, probs = alpha, names = FALSE),
        stats::quantile(x$d1.sims, probs = alpha, names = FALSE),
        stats::quantile(x$z0.sims, probs = alpha, names = FALSE),
        stats::quantile(x$z1.sims, probs = alpha, names = FALSE),
        stats::quantile(x$tau.sims, probs = alpha, names = FALSE),
        stats::quantile(x$n0.sims, probs = alpha, names = FALSE),
        stats::quantile(x$n1.sims, probs = alpha, names = FALSE),
        stats::quantile(x$d.avg.sims, probs = alpha, names = FALSE),
        stats::quantile(x$z.avg.sims, probs = alpha, names = FALSE),
        stats::quantile(x$n.avg.sims, probs = alpha, names = FALSE)
      ),
      stringsAsFactors = FALSE
    )
  }
  out
}


#' @export
ci.margins <- function(x, ci = .95, ...) {
  ci_wald(model = x, ci = ci, dof = Inf, ...)
}


#' @export
ci.lqmm <- function(x, ...) {
  out <- model_parameters(x, ...)
  as.data.frame(out[c("Parameter", "CI_low", "CI_high")])
}

#' @export
ci.lqm <- ci.lqmm


#' @importFrom stats pnorm
#' @export
ci.glht <- function(x, ci = .95, method = "robust", ...) {
  s <- summary(x)
  robust <- !is.null(method) && method == "robust"

  if (robust) {
    adjusted_ci <- 2 * stats::pnorm(s$test$qfunction(ci)) - 1
    dof <- Inf
  } else {
    adjusted_ci <- ci
    dof <- x$df
  }
  out <- ci_wald(model = x, ci = adjusted_ci, dof = dof, ...)

  if (robust) {
    out$CI <- 100 * ci
  }
  out
}


#' @rdname ci.merMod
#' @export
ci.betareg <- function(x, ci = .95, component = c("all", "conditional", "precision"), ...) {
  component <- match.arg(component)
  ci_wald(model = x, ci = ci, dof = Inf, component = component)
}


# ci.vgam <- function(x, ci = .95, component = c("all", "conditional", "smooth"), ...) {
#   component <- match.arg(component)
#
#   # dof and SE
#   dof <- degrees_of_freedom(x)
#   se <- standard_error(x)$SE
#   params <- insight::get_parameters(x)
#
#   se <- se[!is.na(dof)]
#   dof <- dof[!is.na(dof)]
#   params_names <- names(dof)
#
#   # Wald CI for non-chisq parameters
#   out <- ci_wald(model = x, ci = ci, dof = Inf)
#
#   chisq_fac <- stats::qchisq(se, df = dof, lower.tail = FALSE)
#   for (i in 1:length(params_names)) {
#     out$CI_low[out$Parameter == params_names[i]] <- params$Estimate[params$Parameter == params_names[i]] - se[i] * chisq_fac[i]
#     out$CI_high[out$Parameter == params_names[i]] <- params$Estimate[params$Parameter == params_names[i]] + se[i] * chisq_fac[i]
#   }
#
#   out
# }


#' @rdname ci.merMod
#' @export
ci.clm2 <- function(x, ci = .95, component = c("all", "conditional", "scale"), ...) {
  component <- match.arg(component)
  ci_wald(model = x, ci = ci, dof = Inf, component = component)
}

#' @export
ci.clmm2 <- ci.clm2



#' @export
ci.biglm <- function(x, ci = .95, ...) {
  out <- lapply(ci, function(i) {
    ci_list <- stats::confint(x, level = i, ...)
    .data_frame(
      Parameter = rownames(ci_list),
      CI = i * 100,
      CI_low = as.vector(ci_list[, 1]),
      CI_high = as.vector(ci_list[, 2])
    )
  })

  .remove_backticks_from_parameter_names(do.call(rbind, out))
}

#' @export
ci.gls <- ci.biglm


#' @export
ci.lavaan <- function(x, ci = .95, ...) {
  out <- .extract_parameters_lavaan(model = x, ci = ci, ...)
  out$CI <- ci * 100
  out[out$Operator != "~1", c("To", "Operator", "From", "CI", "CI_low", "CI_high")]
}


#' @export
ci.blavaan <- function(x, ci = .95, ...) {
  out <- .extract_parameters_lavaan(model = x, ci = ci, ...)
  out$CI <- ci * 100
  out[out$Operator != "~1", c("To", "Operator", "From", "CI", "CI_low", "CI_high")]
}


#' @rdname ci.merMod
#' @export
ci.lme <- function(x, ci = .95, method = c("wald", "betwithin", "ml1", "satterthwaite"), ...) {
  method <- tolower(method)
  method <- match.arg(method)

  if (method == "wald") {
    if (!requireNamespace("nlme", quietly = TRUE)) {
      ci_wald(model = x, ci = ci)
    } else {
      out <- lapply(ci, function(i) {
        ci_list <- tryCatch(
          {
            nlme::intervals(x, level = i, ...)
          },
          error = function(e) {
            nlme::intervals(x, level = i, which = "fixed", ...)
          }
        )
        .data_frame(
          Parameter = rownames(ci_list$fixed),
          CI = i * 100,
          CI_low = as.vector(ci_list$fixed[, "lower"]),
          CI_high = as.vector(ci_list$fixed[, "upper"])
        )
      })
      .remove_backticks_from_parameter_names(do.call(rbind, out))
    }
    # ml1 approx
  } else if (method == "ml1") {
    ci_ml1(x, ci)

    # betwithin approx
  } else if (method == "betwithin") {
    ci_betwithin(x, ci)

    # Satterthwaite
  } else if (method == "satterthwaite") {
    ci_satterthwaite(x, ci)
  }
}



#' @importFrom insight print_color
#' @importFrom stats qt
#' @export
ci.effectsize_std_params <- function(x, ci = .95, verbose = TRUE, ...) {
  se <- attr(x, "standard_error")

  if (is.null(se)) {
    if (isTRUE(verbose)) {
      insight::print_color("\nCould not extract standard errors of standardized coefficients.\n", "red")
    }
    return(NULL)
  }

  # for "refit" method
  if (is.data.frame(se) && "SE" %in% colnames(se)) {
    se <- se$SE
  }

  # check if we have model. if so, use df from model
  model <- .get_object(x)
  if (!is.null(model)) {
    df <- degrees_of_freedom(model, method = "any")
    if (!is.null(df)) {
      if (length(df) > 1 && length(df) != nrow(x)) {
        df <- Inf
      }
    } else {
      df <- Inf
    }
  } else {
    df <- Inf
  }

  out <- lapply(ci, function(i) {
    alpha <- (1 + i) / 2
    fac <- stats::qt(alpha, df = df)
    data.frame(
      Parameter = x$Parameter,
      CI = i * 100,
      CI_low = x$Std_Coefficient - se * fac,
      CI_high = x$Std_Coefficient + se * fac,
      stringsAsFactors = FALSE
    )
  })

  .remove_backticks_from_parameter_names(do.call(rbind, out))
}

#' @export
ci.effectsize_table <- ci.effectsize_std_params


#' @export
ci.rma <- function(x, ci = .95, ...) {
  params <- insight::get_parameters(x)
  out <- tryCatch(
    {
      tmp <- lapply(ci, function(i) {
        model <- stats::update(x, level = i)
        .data_frame(
          Parameter = params$Parameter,
          CI = i * 100,
          CI_low = as.vector(model$ci.lb),
          CI_high = as.vector(model$ci.ub)
        )
      })
      .remove_backticks_from_parameter_names(do.call(rbind, tmp))
    },
    error = function(e) {
      NULL
    }
  )
  if (is.null(out)) {
    se <- standard_error(x)
    out <- lapply(ci, function(i) {
      alpha <- (1 + i) / 2
      fac <- stats::qnorm(alpha)
      .data_frame(
        Parameter = params$Parameter,
        CI = i * 100,
        CI_low = params$Estimate - as.vector(se$SE) * fac,
        CI_high = params$Estimate + as.vector(se$SE) * fac
      )
    })
    out <- .remove_backticks_from_parameter_names(do.call(rbind, out))
  }
  out
}



#' @export
ci.metaplus <- function(x, ...) {
  out <- .data_frame(
    Parameter = .remove_backticks_from_string(rownames(x$results)),
    CI_low = as.vector(x$results[, "95% ci.lb"]),
    CI_high = as.vector(x$results[, "95% ci.ub"])
  )

  out$Parameter[grepl("muhat", out$Parameter)] <- "(Intercept)"
  out
}


#' @export
ci.meta_random <- function(x, method = "hdi", ...) {
  # process arguments
  params <- as.data.frame(x$estimates)
  ci_method <- match.arg(method, choices = c("hdi", "eti"))

  # extract ci-level and find ci-columns
  ci <- .meta_bma_extract_ci(params)
  ci_cols <- .metabma_ci_columns(ci_method, ci)

  out <- data.frame(
    Parameter = rownames(params),
    CI = .95,
    CI_low = params[[ci_cols[1]]],
    CI_high = params[[ci_cols[2]]],
    stringsAsFactors = FALSE
  )

  out$Parameter[grepl("d", out$Parameter)] <- "(Intercept)"
  out
}

#' @export
ci.meta_fixed <- ci.meta_random

#' @export
ci.meta_bma <- ci.meta_random





# helper -----------------------------------------


#' @keywords internal
.check_component <- function(m, x, verbose = TRUE) {
  if (!insight::model_info(m)$is_zero_inflated && x %in% c("zi", "zero_inflated")) {
    if (isTRUE(verbose)) {
      insight::print_color("Model has no zero-inflation component!\n", "red")
    }
    x <- NULL
  }
  x
}
