% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/oem_xval.R
\name{xval.oem}
\alias{xval.oem}
\title{Fast cross validation for Orthogonalizing EM}
\usage{
xval.oem(x, y, nfolds = 10L, foldid = NULL, type.measure = c("mse",
  "deviance", "class", "auc", "mae"), ncores = -1,
  family = c("gaussian", "binomial"), penalty = c("elastic.net",
  "lasso", "ols", "mcp", "scad", "mcp.net", "scad.net", "grp.lasso",
  "grp.lasso.net", "grp.mcp", "grp.scad", "grp.mcp.net", "grp.scad.net",
  "sparse.grp.lasso"), weights = numeric(0), lambda = numeric(0),
  nlambda = 100L, lambda.min.ratio = NULL, alpha = 1, gamma = 3,
  tau = 0.5, groups = numeric(0), penalty.factor = NULL,
  group.weights = NULL, standardize = TRUE, intercept = TRUE,
  maxit = 500L, tol = 1e-07, irls.maxit = 100L, irls.tol = 0.001,
  compute.loss = FALSE)
}
\arguments{
\item{x}{input matrix of dimension n x p (sparse matrices not yet implemented). 
Each row is an observation, each column corresponds to a covariate. The xval.oem() function
is optimized for n >> p settings and may be very slow when p > n, so please use other packages
such as \code{glmnet}, \code{ncvreg}, \code{grpreg}, or \code{gglasso} when p > n or p approx n.}

\item{y}{numeric response vector of length \code{nobs = nrow(x)}.}

\item{nfolds}{integer number of cross validation folds. 3 is the minimum number allowed. defaults to 10}

\item{foldid}{an optional vector of values between 1 and \code{nfold} specifying which fold each observation belongs to.}

\item{type.measure}{measure to evaluate for cross-validation. The default is \code{type.measure = "deviance"}, 
which uses squared-error for gaussian models (a.k.a \code{type.measure = "mse"} there), deviance for logistic
regression. \code{type.measure = "class"} applies to \code{binomial} only. \code{type.measure = "auc"} is for two-class logistic 
regression only. \code{type.measure="mse"} or \code{type.measure="mae"} (mean absolute error) can be used by all models;
they measure the deviation from the fitted mean to the response.}

\item{ncores}{Integer scalar that specifies the number of threads to be used}

\item{family}{\code{"gaussian"} for least squares problems, \code{"binomial"} for binary response (not implemented yet).}

\item{penalty}{Specification of penalty type. Choices include:
\itemize{
   \item{\code{"elastic.net"}}{ - elastic net penalty, extra parameters: \code{"alpha"}}
   \item{\code{"lasso"}}{ - lasso penalty}
   \item{\code{"ols"}}{ - ordinary least squares}
   \item{\code{"mcp"}}{ - minimax concave penalty, extra parameters: \code{"gamma"}}
   \item{\code{"scad"}}{ - smoothly clipped absolute deviation, extra parameters: \code{"gamma"}}
   \item{\code{"mcp.net"}}{ - minimax concave penalty + l2 penalty, extra parameters: 
   \code{"gamma"}, \code{"alpha"}}
   \item{\code{"scad.net"}}{ - smoothly clipped absolute deviation + l2 penalty, extra parameters: 
   \code{"gamma"}, \code{"alpha"}}
   \item{\code{"grp.lasso"}}{ - group lasso penalty}
   \item{\code{"grp.lasso.net"}}{ - group lasso penalty + l2 penalty, extra parameters: \code{"alpha"}}
   \item{\code{"grp.mcp"}}{ - group minimax concave penalty, extra parameters: \code{"gamma"}}
   \item{\code{"grp.scad"}}{ - group smoothly clipped absolute deviation, extra parameters: \code{"gamma"}}
   \item{\code{"grp.mcp.net"}}{ - group minimax concave penalty + l2 penalty, extra parameters: \code{"gamma"}, \code{"alpha"}}
   \item{\code{"grp.scad.net"}}{ - group smoothly clipped absolute deviation + l2 penalty, extra parameters: \code{"gamma"}, \code{"alpha"}}
   \item{\code{"sparse.grp.lasso"}}{ - sparse group lasso penalty (group lasso + lasso), extra parameters: \code{"tau"}}
}
Careful consideration is required for the group lasso, group MCP, and group SCAD penalties. Groups as specified by the \code{groups} argument 
should be chosen in a sensible manner.}

\item{weights}{observation weights. defaults to 1 for each observation (setting weight vector to 
length 0 will default all weights to 1)}

\item{lambda}{A user supplied lambda sequence. By default, the program computes
its own lambda sequence based on \code{nlambda} and \code{lambda.min.ratio}. Supplying
a value of lambda overrides this.}

\item{nlambda}{The number of lambda values - default is 100.}

\item{lambda.min.ratio}{Smallest value for lambda, as a fraction of \code{lambda.max}, the (data derived) entry
value (i.e. the smallest value for which all coefficients are zero). The default
depends on the sample size nobs relative to the number of variables nvars. If
\code{nobs > nvars}, the default is 0.0001, close to zero.}

\item{alpha}{mixing value for \code{elastic.net}, \code{mcp.net}, \code{scad.net}, \code{grp.mcp.net}, \code{grp.scad.net}. 
penalty applied is (1 - alpha) * (ridge penalty) + alpha * (lasso/mcp/mcp/grp.lasso penalty)}

\item{gamma}{tuning parameter for SCAD and MCP penalties. must be >= 1}

\item{tau}{mixing value for \code{sparse.grp.lasso}. penalty applied is (1 - tau) * (group lasso penalty) + tau * (lasso penalty)}

\item{groups}{A vector of describing the grouping of the coefficients. See the example below. All unpenalized variables
should be put in group 0}

\item{penalty.factor}{Separate penalty factors can be applied to each coefficient. 
This is a number that multiplies lambda to allow differential shrinkage. Can be 0 for some variables, 
which implies no shrinkage, and that variable is always included in the model. Default is 1 for all 
variables.}

\item{group.weights}{penalty factors applied to each group for the group lasso. Similar to \code{penalty.factor}, 
this is a number that multiplies lambda to allow differential shrinkage. Can be 0 for some groups, 
which implies no shrinkage, and that group is always included in the model. Default is sqrt(group size) for all
groups.}

\item{standardize}{Logical flag for \code{x} variable standardization, prior to fitting the models. 
The coefficients are always returned on the original scale. Default is \code{standardize = TRUE}. If 
variables are in the same units already, you might not wish to standardize.}

\item{intercept}{Should intercept(s) be fitted (\code{default = TRUE}) or set to zero (\code{FALSE})}

\item{maxit}{integer. Maximum number of OEM iterations}

\item{tol}{convergence tolerance for OEM iterations}

\item{irls.maxit}{integer. Maximum number of IRLS iterations}

\item{irls.tol}{convergence tolerance for IRLS iterations. Only used if \code{family != "gaussian"}}

\item{compute.loss}{should the loss be computed for each estimated tuning parameter? Defaults to \code{FALSE}. Setting
to \code{TRUE} will dramatically increase computational time}
}
\value{
An object with S3 class \code{"xval.oem"}
}
\description{
Fast cross validation for Orthogonalizing EM
}
\examples{
set.seed(123)
n.obs <- 1e4
n.vars <- 100

true.beta <- c(runif(15, -0.25, 0.25), rep(0, n.vars - 15))

x <- matrix(rnorm(n.obs * n.vars), n.obs, n.vars)
y <- rnorm(n.obs, sd = 3) + x \%*\% true.beta

system.time(fit <- oem(x = x, y = y, 
                       penalty = c("lasso", "grp.lasso"), 
                       groups = rep(1:20, each = 5)))
                       
system.time(xfit <- xval.oem(x = x, y = y, 
                             penalty = c("lasso", "grp.lasso"), 
                             groups = rep(1:20, each = 5)))
                             
system.time(xfit2 <- xval.oem(x = x, y = y, 
                              penalty = c("lasso", "grp.lasso",
                                          "mcp",       "scad", 
                                          "mcp.net",   "scad.net",
                                          "grp.lasso", "grp.lasso.net",
                                          "grp.mcp",   "grp.scad",
                                          "sparse.grp.lasso"), 
                              groups = rep(1:20, each = 5)))

}
