% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/adp.R
\docType{class}
\name{adp-class}
\alias{adp-class}
\title{Class to Store adp (ADCP) Data}
\description{
This class stores data from acoustic Doppler profilers. Some manufacturers
call these ADCPs, while others call them ADPs; here the shorter form is
used by analogy to ADVs.
}
\section{Slots}{

\describe{
\item{\code{data}}{As with all \code{oce} objects, the \code{data} slot
for \code{adp} objects is a
\link{list} containing the main data for the object.
The key items stored in this slot include \code{time}, \code{distance}, and \code{v}, along with angles \code{heading}, \code{pitch} and \code{roll}.}

\item{\code{metadata}}{As with all \code{oce} objects, the \code{metadata} slot
for \code{adp} objects is a \link{list} containing
information about the \code{data} or about the object itself.
Examples that are of common interest include \code{oceCoordinate}, \code{orientation}, \code{frequency}, and \code{beamAngle}.}

\item{\code{processingLog}}{As with all \code{oce} objects, the \code{processingLog} slot
for \code{adp} objects is a
\link{list} with entries describing the creation and evolution
of the object. The contents are updated by various \code{oce} functions to
keep a record of processing steps.  Object summaries and
\code{\link[=processingLogShow]{processingLogShow()}} both display the log.}
}}

\section{Modifying slot contents}{


Although the \code{[[<-} operator may permit modification of the contents
of \linkS4class{adp} objects (see \code{\link{[[<-,adp-method}}),
it is better to use \code{\link[=oceSetData]{oceSetData()}} and \code{\link[=oceSetMetadata]{oceSetMetadata()}},
because those functions save an entry in the \code{processingLog}
that describes the change.
}

\section{Retrieving slot contents}{


The full contents of the \code{data} and \code{metadata} slots of a \linkS4class{adp}
object may be retrieved in the standard R way using \code{\link[=slot]{slot()}}. For
example \code{slot(o,"data")} returns the \code{data} slot of an object named \code{o},
and similarly \code{slot(o,"metadata")} returns
the \code{metadata} slot.

The slots may also be obtained with the \code{\link{[[,adp-method}}
operator, as e.g. \code{o[["data"]]} and \code{o[["metadata"]]}, respectively.

The \code{\link{[[,adp-method}} operator can also
be used to retrieve items from within the \code{data} and \code{metadata} slots.
For example, \code{o[["temperature"]]} can be used to retrieve temperature
from an object containing that quantity. The rule is that a named
quantity is sought first within the object's \code{metadata} slot,
with the \code{data} slot being checked only if \code{metadata} does not
contain the item. This \code{[[} method can also be used to get
certain derived quantities, if the object contains sufficient
information to calculate them. For example, an object that holds
(practical) salinity, temperature and pressure, along with
longitude and latitude, has sufficient information to compute
Absolute Salinity, and so \code{o[["SA"]]} will yield the
calculated Absolute Salinity.

It is also possible to find items more directly, using \code{\link[=oceGetData]{oceGetData()}} and
\code{\link[=oceGetMetadata]{oceGetMetadata()}}, but neither of these functions can
retrieve derived items.
}

\section{Reading/creating \code{adp} objects}{


The \code{metadata} slot contains various
items relating to the dataset, including source file name, sampling rate,
velocity resolution, velocity maximum value, and so on.  Some of these are
particular to particular instrument types, and prudent researchers will take
a moment to examine the whole contents of the metadata, either in summary
form (with \code{str(adp[["metadata"]])}) or in detail (with
\code{adp[["metadata"]]}).  Perhaps the most useful general properties are
\code{adp[["bin1Distance"]]} (the distance, in metres, from the sensor to
the bottom of the first bin), \code{adp[["cellSize"]]} (the cell height, in
metres, in the vertical direction, \emph{not} along the beam), and
\code{adp[["beamAngle"]]} (the angle, in degrees, between beams and an
imaginary centre line that bisects all beam pairs).

The diagram provided below indicates the coordinate-axis and beam-numbering
conventions for three- and four-beam ADP devices, viewed as though the
reader were looking towards the beams being emitted from the transducers.

\if{html}{\figure{adp_beams.png}{options: width="400" alt="Figure: adp_beams.png"}}

The bin geometry of a four-beam profiler is illustrated below, for
\code{adp[["beamAngle"]]} equal to 20 degrees, \code{adp[["bin1Distance"]]}
equal to 2m, and \code{adp[["cellSize"]]} equal to 1m.   In the diagram, the
viewer is in the plane containing two beams that are not shown, so the two
visible beams are separated by 40 degrees.  Circles indicate the centres of
the range-gated bins within the beams.  The lines enclosing those circles
indicate the coverage of beams that spread plus and minus 2.5 degrees from
their centreline.

\if{html}{\figure{adpgeometry2.png}{options: width="400" alt="Figure: adpgeometry2.png"}}

Note that \code{adp[["oceCoordinate"]]} stores the present coordinate system
of the object, and it has possible values \code{"beam"}, \code{"xyz"}, \code{"sfm"} or
\code{"enu"}.  (This should not be confused with
\code{adp[["originalCoordinate"]]}, which stores the coordinate system used
in the original data file.)

The \code{data} slot holds some standardized items, and
many that vary from instrument to instrument.  One standard item is
\code{adp[["v"]]}, a three-dimensional numeric array of velocities in
m/s.  In this matrix, the first index indicates time, the second bin
number, and the third beam number. The meaning of beams number depends on
whether the object is in beam coordinates, frame coordinates, or earth
coordinates.  For example, if in earth coordinates, then beam 1 is
the eastward component of velocity.
Thus, for example,
\preformatted{
library(oce)
data(adp)
t <- adp[["time"]]
d <- adp[["distance"]]
eastward <- adp[["v"]][,,1]
imagep(t, d, eastward, missingColor="gray")
}
plots an image of the eastward component of velocity as a function of time (the x axis)
and distance from sensor (y axis), since the \code{adp} dataset is
in earth coordinates. Note the semidurnal tidal signal, and the pattern of missing
data at the ocean surface (gray blotches at the top).

Corresponding to the velocity array are two arrays of type raw, and
identical dimension, accessed by \code{adp[["a"]]} and \code{adp[["q"]]},
holding measures of signal strength and data quality (referred
to as "correlation" in some documentation),
respectively.  (The exact meanings of these depend on the particular type
of instrument, and it is assumed that users will be familiar enough with
instruments to know both the meanings and their practical consequences in
terms of data-quality assessment, etc.)

In addition to the arrays, there are time-based vectors.  The vector
\code{adp[["time"]]} (of length equal to the first index of
\code{adp[["v"]]}, etc.) holds times of observation.  Depending on type of
instrument and its configuration, there may also be corresponding vectors
for sound speed (\code{adp[["soundSpeed"]]}), pressure
(\code{adp[["pressure"]]}), temperature (\code{adp[["temperature"]]}),
heading (\code{adp[["heading"]]}) pitch (\code{adp[["pitch"]]}), and roll
(\code{adp[["roll"]]}), depending on the setup of the instrument.

The precise meanings of the data items depend on the instrument type.  All
instruments have \code{v} (for velocity), \code{q} (for a measure of data
quality) and \code{a} (for a measure of backscatter amplitude, also called
echo intensity).
Teledyne-RDI profilers have an additional item \code{g} (for
percent-good).

VmDas-equipped Teledyne-RDI profilers additional navigation data, with
details listed in the table below; note that the RDI documentation (reference 2) and
the RDI gui use inconsistent names for most items.

\tabular{lll}{
\strong{Oce name}\tab \strong{RDI doc name}\tab \strong{RDI GUI name}\cr
\code{avgSpeed}\tab Avg Speed\tab Speed/Avg/Mag\cr
\code{avgMagnitudeVelocityEast}\tab Avg Mag Vel East\tab ?\cr
\code{avgMagnitudeVelocityNorth}\tab Avg Mag Vel North\tab ?\cr
\code{avgTrackMagnetic}\tab Avg Track Magnetic\tab Speed/Avg/Dir (?)\cr
\code{avgTrackTrue}\tab Avg Track True\tab Speed/Avg/Dir (?)\cr
\code{avgTrueVelocityEast}\tab Avg True Vel East\tab ?\cr
\code{avgTrueVelocityNorth}\tab Avg True Vel North\tab ?\cr
\code{directionMadeGood}\tab Direction Made Good\tab Speed/Made Good/Dir\cr
\code{firstLatitude}\tab First latitude\tab Start Lat\cr
\code{firstLongitude}\tab First longitude\tab Start Lon\cr
\code{firstTime}\tab UTC Time of last fix\tab End Time\cr
\code{lastLatitude}\tab Last latitude\tab End Lat\cr
\code{lastLongitude}\tab Last longitude\tab End Lon\cr
\code{lastTime}\tab UTC Time of last fix\tab End Time\cr
\code{numberOfHeadingSamplesAveraged}\tab Number heading samples averaged\tab ?\cr
\code{numberOfMagneticTrackSamplesAveraged}\tab Number of magnetic track samples averaged\tab ? \cr
\code{numberOfPitchRollSamplesAvg}\tab Number of magnetic track samples averaged\tab ? \cr
\code{numberOfSpeedSamplesAveraged}\tab Number of speed samples averaged\tab ? \cr
\code{numberOfTrueTrackSamplesAvg}\tab Number of true track samples averaged\tab ? \cr
\code{primaryFlags}\tab Primary Flags\tab ?\cr
\code{shipHeading}\tab Heading\tab ?\cr
\code{shipPitch}\tab Pitch\tab ?\cr
\code{shipRoll}\tab Roll\tab ?\cr
\code{speedMadeGood}\tab Speed Made Good\tab Speed/Made Good/Mag\cr
\code{speedMadeGoodEast}\tab Speed MG East\tab ?\cr
\code{speedMadeGoodNorth}\tab Speed MG North\tab ?\cr
}

For Teledyne-RDI profilers, there are four three-dimensional arrays
holding beamwise data.  In these, the first index indicates time, the
second bin number, and the third beam number (or coordinate number, for
data in \code{xyz}, \code{sfm}, \code{enu} or \code{other} coordinate systems).  In
the list below, the quoted phrases are quantities as defined in Figure 9
of reference 1.
\itemize{
\item \code{v} is \code{velocity} in m/s, inferred from two-byte signed
integer values (multiplied by the scale factor that is stored in
\code{velocityScale} in the metadata).
\item \code{q} is "correlation magnitude" a one-byte quantity stored
as type \code{raw} in the object. The values may range from 0 to 255.
\item \code{a} is "backscatter amplitude", also known as "echo
intensity" a one-byte quantity stored as type \code{raw} in the object.
The values may range from 0 to 255.
\item \code{g} is "percent good" a one-byte quantity stored as \code{raw}
in the object.  The values may range from 0 to 100.
}

Finally, there is a vector \code{adp[["distance"]]} that indicates the bin
distances from the sensor, measured in metres along an imaginary centre
line bisecting beam pairs.  The length of this vector equals
\code{dim(adp[["v"]])[2]}.
}

\section{Teledyne-RDI Sentinel V ADCPs}{
 As of 2016-09-27 there is
provisional support for the TRDI "SentinelV" ADCPs, which are 5
beam ADCPs with a vertical centre beam. Relevant vertical beam
fields are called \code{adp[["vv"]]}, \code{adp[["va"]]},
\code{adp[["vq"]]}, and \code{adp[["vg"]]} in analogy with the
standard 4-beam fields.
}

\section{Accessing and altering information within \linkS4class{adp} objects}{

\emph{Extracting values} Matrix data may be accessed as illustrated
above, e.g.  or an adp object named \code{adv}, the data are provided by
\code{adp[["v"]]}, \code{adp[["a"]]}, and \code{adp[["q"]]}.  As a
convenience, the last two of these can be accessed as numeric (as opposed to
raw) values by e.g.  \code{adp[["a", "numeric"]]}.  The vectors are accessed
in a similar way, e.g. \code{adp[["heading"]]}, etc.  Quantities in the
\code{metadata} slot are also available by name, e.g.
\code{adp[["velocityResolution"]]}, etc.

\emph{Assigning values.} This follows the standard form, e.g. to increase
all velocity data by 1 cm/s, use \code{adp[["v"]] <- 0.01 + adp[["v"]]}.

\emph{Overview of contents} The \code{show} method (e.g.
\code{show(d)}) displays information about an ADP object named \code{d}.
}

\section{Dealing with suspect data}{

There are many possibilities for confusion
with \code{adp} devices, owing partly to the flexibility that manufacturers
provide in the setup.  Prudent users will undertake many tests before trusting
the details of the data.  Are mean currents in the expected direction, and of
the expected magnitude, based on other observations or physical constraints?
Is the phasing of currents as expected?  If the signals are suspect, could an
incorrect scale account for it?  Could the transformation matrix be incorrect?
Might the data have exceeded the maximum value, and then ``wrapped around'' to
smaller values?  Time spent on building confidence in data quality is seldom
time wasted.
}

\section{References}{

\enumerate{
\item Teledyne-RDI, 2007.
\emph{WorkHorse commands and output data format.}
P/N 957-6156-00 (November 2007).
\item Teledyne-RDI, 2012. \emph{VmDas User's Guide, Ver. 1.46.5}.
}
}

\seealso{
A file containing ADP data is usually recognized by Oce, and so
\code{\link[=read.oce]{read.oce()}} will usually read the data.  If not, one may use the
general ADP function \code{\link[=read.adp]{read.adp()}} or specialized variants
\code{\link[=read.adp.rdi]{read.adp.rdi()}}, \code{\link[=read.adp.nortek]{read.adp.nortek()}},
\code{\link[=read.adp.ad2cp]{read.adp.ad2cp()}},
\code{\link[=read.adp.sontek]{read.adp.sontek()}} or \code{\link[=read.adp.sontek.serial]{read.adp.sontek.serial()}}.

ADP data may be plotted with \code{\link[=plot,adp-method]{plot,adp-method()}}, which is a
generic function so it may be called simply as \code{plot}.

Statistical summaries of ADP data are provided by the generic function
\code{summary}, while briefer overviews are provided with \code{show}.

Conversion from beam to xyz coordinates may be done with
\code{\link[=beamToXyzAdp]{beamToXyzAdp()}}, and from xyz to enu (east north up) may be done
with \code{\link[=xyzToEnuAdp]{xyzToEnuAdp()}}.  \code{\link[=toEnuAdp]{toEnuAdp()}} may be used to
transfer either beam or xyz to enu.  Enu may be converted to other coordinates
(e.g. aligned with a coastline) with \code{\link[=enuToOtherAdp]{enuToOtherAdp()}}.

Other classes provided by oce: 
\code{\link{adv-class}},
\code{\link{argo-class}},
\code{\link{bremen-class}},
\code{\link{cm-class}},
\code{\link{coastline-class}},
\code{\link{ctd-class}},
\code{\link{lisst-class}},
\code{\link{lobo-class}},
\code{\link{met-class}},
\code{\link{oce-class}},
\code{\link{odf-class}},
\code{\link{rsk-class}},
\code{\link{sealevel-class}},
\code{\link{section-class}},
\code{\link{topo-class}},
\code{\link{windrose-class}},
\code{\link{xbt-class}}

Other things related to adp data: 
\code{\link{[[,adp-method}},
\code{\link{[[<-,adp-method}},
\code{\link{ad2cpCodeToName}()},
\code{\link{ad2cpHeaderValue}()},
\code{\link{adpAd2cpFileTrim}()},
\code{\link{adpConvertRawToNumeric}()},
\code{\link{adpEnsembleAverage}()},
\code{\link{adpFlagPastBoundary}()},
\code{\link{adpRdiFileTrim}()},
\code{\link{adp_rdi.000}},
\code{\link{adp}},
\code{\link{applyMagneticDeclination,adp-method}},
\code{\link{as.adp}()},
\code{\link{beamName}()},
\code{\link{beamToXyzAdpAD2CP}()},
\code{\link{beamToXyzAdp}()},
\code{\link{beamToXyzAdv}()},
\code{\link{beamToXyz}()},
\code{\link{beamUnspreadAdp}()},
\code{\link{binmapAdp}()},
\code{\link{enuToOtherAdp}()},
\code{\link{enuToOther}()},
\code{\link{handleFlags,adp-method}},
\code{\link{is.ad2cp}()},
\code{\link{plot,adp-method}},
\code{\link{read.adp.ad2cp}()},
\code{\link{read.adp.nortek}()},
\code{\link{read.adp.rdi}()},
\code{\link{read.adp.sontek.serial}()},
\code{\link{read.adp.sontek}()},
\code{\link{read.adp}()},
\code{\link{read.aquadoppHR}()},
\code{\link{read.aquadoppProfiler}()},
\code{\link{read.aquadopp}()},
\code{\link{rotateAboutZ}()},
\code{\link{setFlags,adp-method}},
\code{\link{subset,adp-method}},
\code{\link{subtractBottomVelocity}()},
\code{\link{summary,adp-method}},
\code{\link{toEnuAdp}()},
\code{\link{toEnu}()},
\code{\link{velocityStatistics}()},
\code{\link{xyzToEnuAdpAD2CP}()},
\code{\link{xyzToEnuAdp}()},
\code{\link{xyzToEnu}()}
}
\concept{classes provided by oce}
\concept{things related to adp data}
