% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stats.R
\name{classify_adopters}
\alias{classify_adopters}
\alias{classify}
\alias{classify_adopters.diffnet}
\alias{classify_adopters.default}
\alias{ftable.diffnet_adopters}
\alias{as.data.frame.diffnet_adopters}
\alias{plot.diffnet_adopters}
\title{Classify adopters accordingly to Time of Adoption and Threshold levels.}
\usage{
classify_adopters(...)

classify(...)

\method{classify_adopters}{diffnet}(graph, include_censored = FALSE, ...)

\method{classify_adopters}{default}(
  graph,
  toa,
  t0 = NULL,
  t1 = NULL,
  expo = NULL,
  include_censored = FALSE,
  ...
)

\method{ftable}{diffnet_adopters}(x, as.pcent = TRUE, digits = 2, ...)

\method{as.data.frame}{diffnet_adopters}(x, row.names = NULL, optional = FALSE, ...)

\method{plot}{diffnet_adopters}(x, y = NULL, ftable.args = list(), table.args = list(), ...)
}
\arguments{
\item{...}{Further arguments passed to the method.}

\item{graph}{A dynamic graph.}

\item{include_censored}{Logical scalar, passed to \code{\link{threshold}}.}

\item{toa}{Integer vector of length \eqn{n} with times of adoption.}

\item{t0}{Integer scalar passed to \code{\link{threshold}} and \code{\link{toa_mat}}.}

\item{t1}{Integer scalar passed to \code{\link{toa_mat}}.}

\item{expo}{Numeric matrix of size \eqn{n\times T}{n*T} with network exposures.}

\item{x}{A \code{diffnet_adopters} class object.}

\item{as.pcent}{Logical scalar. When \code{TRUE} returns a table with percentages
instead.}

\item{digits}{Integer scalar. Passed to \code{\link[base:round]{round}}.}

\item{row.names}{Passed to \code{\link[base:as.data.frame]{as.data.frame}}.}

\item{optional}{Passed to \code{\link[base:as.data.frame]{as.data.frame}}.}

\item{y}{Ignored.}

\item{ftable.args}{List of arguments passed to \code{\link[stats:ftable]{ftable}}.}

\item{table.args}{List of arguments passed to \code{\link{table}}.}
}
\value{
A list of class \code{diffnet_adopters} with the following elements:
\item{toa}{A factor vector of length \eqn{n} with 4 levels:
 "Early Adopters", "Early Majority", "Late Majority", and "Laggards"}
\item{thr}{A factor vector of length \eqn{n} with 4 levels:
 "Very Low Thresh.", "Low Thresh.", "High Thresh.", and "Very High Thresh."}
}
\description{
Adopters are classified as in Valente (1995). In general, this is done
depending on the distance in terms of standard deviations from the mean of
Time of Adoption and Threshold.
}
\details{
Classifies (only) adopters according to time of adoption and threshold as
described in Valente (1995). In particular, the categories are defined as follow:

For Time of Adoption, with \code{toa} as the vector of times of adoption:
\itemize{
 \item \emph{Early Adopters}: \code{toa[i] <= mean(toa) - sd(toa)},
 \item \emph{Early Majority}: \code{mean(toa) - sd(toa) < toa[i] <= mean(toa) },
 \item \emph{Late Majority}: \code{mean(toa) < toa[i] <= mean(toa) + sd(toa) }, and
 \item \emph{Laggards}: \code{mean(toa) + sd(toa) < toa[i] }.
}

For Threshold levels, with \code{thr} as the vector of threshold levels:
\itemize{
 \item \emph{Very Low Thresh.}: \code{thr[i] <= mean(thr) - sd(thr)},
 \item \emph{Low Thresh.}: \code{mean(thr) - sd(thr) < thr[i] <= mean(thr) },
 \item \emph{High Thresh.}: \code{mean(thr) < thr[i] <= mean(thr) + sd(thr) }, and
 \item \emph{Very High. Thresh.}: \code{mean(thr) + sd(thr) < thr[i] }.
}

By default threshold levels are not computed for left censored data. These
will have a \code{NA} value in the \code{thr} vector.

The plot method, \code{plot.diffnet_adopters}, is a wrapper for the
\code{\link[graphics:plot.table]{plot.table}} method. This generates a
\code{\link[graphics:mosaicplot]{mosaicplot}} plot.
}
\examples{
# Classifying brfarmers -----------------------------------------------------

x <- brfarmersDiffNet
diffnet.toa(x)[x$toa==max(x$toa, na.rm = TRUE)] <- NA
out <- classify_adopters(x)

# This is one way
round(
with(out, ftable(toa, thr, dnn=c("Time of Adoption", "Threshold")))/
  nnodes(x[!is.na(x$toa)])*100, digits=2)

# This is other
ftable(out)

# Can be coerced into a data.frame, e.g. ------------------------------------
 str(classify(brfarmersDiffNet))
 ans <- cbind(
 as.data.frame(classify(brfarmersDiffNet)), brfarmersDiffNet$toa
 )
 head(ans)

# Creating a mosaic plot with the medical innovations -----------------------
x <- classify(medInnovationsDiffNet)
plot(x)

}
\references{
Valente, T. W. (1995). "Network models of the diffusion of innovations"
 (2nd ed.). Cresskill N.J.: Hampton Press.
}
\seealso{
Other statistics: 
\code{\link{bass}},
\code{\link{cumulative_adopt_count}()},
\code{\link{dgr}()},
\code{\link{ego_variance}()},
\code{\link{exposure}()},
\code{\link{hazard_rate}()},
\code{\link{infection}()},
\code{\link{moran}()},
\code{\link{struct_equiv}()},
\code{\link{threshold}()},
\code{\link{vertex_covariate_dist}()}
}
\author{
George G. Vega Yon
}
\concept{statistics}
