% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/outercv.R
\name{outercv}
\alias{outercv}
\alias{outercv.default}
\alias{outercv.formula}
\title{Outer cross-validation of selected models}
\usage{
outercv(y, ...)

\method{outercv}{default}(
  y,
  x,
  model,
  filterFUN = NULL,
  filter_options = NULL,
  weights = NULL,
  balance = NULL,
  balance_options = NULL,
  modifyX = NULL,
  modifyX_useY = FALSE,
  modifyX_options = NULL,
  outer_method = c("cv", "LOOCV"),
  n_outer_folds = 10,
  outer_folds = NULL,
  cv.cores = 1,
  multicore_fork = (Sys.info()["sysname"] != "Windows"),
  predict_type = "prob",
  outer_train_predict = FALSE,
  na.option = "pass",
  returnList = FALSE,
  verbose = FALSE,
  suppressMsg = verbose,
  ...
)

\method{outercv}{formula}(
  formula,
  data,
  model,
  outer_method = c("cv", "LOOCV"),
  n_outer_folds = 10,
  outer_folds = NULL,
  cv.cores = 1,
  multicore_fork = (Sys.info()["sysname"] != "Windows"),
  predict_type = "prob",
  outer_train_predict = FALSE,
  verbose = FALSE,
  suppressMsg = verbose,
  ...,
  na.action = na.fail
)
}
\arguments{
\item{y}{Response vector}

\item{...}{Optional arguments passed to the function specified by \code{model}.}

\item{x}{Matrix or dataframe of predictors}

\item{model}{Character value or function of the model to be fitted.}

\item{filterFUN}{Filter function, e.g. \link{ttest_filter} or \link{relieff_filter}.
Any function can be provided and is passed \code{y} and \code{x}. Must return a
character vector with names of filtered predictors. Not available if
\code{outercv} is called with a formula.}

\item{filter_options}{List of additional arguments passed to the filter
function specified by \code{filterFUN}.}

\item{weights}{Weights applied to each sample for models which can use
weights. Note \code{weights} and \code{balance} cannot be used at the same time.
Weights are not applied in filters.}

\item{balance}{Specifies method for dealing with imbalanced class data.
Current options are \code{"randomsample"} or \code{"smote"}. Not available if
\code{outercv} is called with a formula. See \code{\link[=randomsample]{randomsample()}} and \code{\link[=smote]{smote()}}}

\item{balance_options}{List of additional arguments passed to the balancing
function}

\item{modifyX}{Character string specifying the name of a function to modify
\code{x}. This can be an imputation function for replacing missing values, or a
more complex function which alters or even adds columns to \code{x}. The
required return value of this function depends on the \code{modifyX_useY}
setting.}

\item{modifyX_useY}{Logical value whether the \code{x} modifying function makes
use of response training data from \code{y}. If \code{FALSE} then the \code{modifyX}
function simply needs to return a modified \code{x} object. If \code{TRUE} then the
\code{modifyX} function must return a model type object on which \code{predict()} can
be called, so that train and test partitions of \code{x} can be modified
independently.}

\item{modifyX_options}{List of additional arguments passed to the \code{x}
modifying function}

\item{outer_method}{String of either \code{"cv"} or \code{"LOOCV"} specifying whether
to do k-fold CV or leave one out CV (LOOCV) for the outer folds}

\item{n_outer_folds}{Number of outer CV folds}

\item{outer_folds}{Optional list containing indices of test folds for outer
CV. If supplied, \code{n_outer_folds} is ignored.}

\item{cv.cores}{Number of cores for parallel processing of the outer loops.}

\item{multicore_fork}{Logical whether to use forked multicore parallel
processing. Forked multicore processing uses \code{parallel::mclapply}. It is
only available on unix/mac as windows does not allow forking. It is set to
\code{FALSE} by default in windows and \code{TRUE} in unix/mac. Non-forked parallel
processing is executed using \code{parallel::parLapply} or \code{pbapply::pblapply}
if \code{verbose} is \code{TRUE}.}

\item{predict_type}{Only used with binary classification. Calculation of ROC
AUC requires predicted class probabilities from fitted models. Most model
functions use syntax of the form \code{predict(..., type = "prob")}. However,
some models require a different \code{type} to be specified, which can be passed
to \code{predict()} via \code{predict_type}.}

\item{outer_train_predict}{Logical whether to save predictions on outer
training folds to calculate performance on outer training folds.}

\item{na.option}{Character value specifying how \code{NA}s are dealt with.
\code{"omit"} is equivalent to \code{na.action = na.omit}. \code{"omitcol"} removes cases
if there are \code{NA} in 'y', but columns (predictors) containing \code{NA} are
removed from 'x' to preserve cases. Any other value means that \code{NA} are
ignored (a message is given).}

\item{returnList}{Logical whether to return list of results after main outer
CV loop without concatenating results. Useful for debugging.}

\item{verbose}{Logical whether to print messages and show progress}

\item{suppressMsg}{Logical whether to suppress messages and printed output
from model functions. This is necessary when using forked multicore
parallelisation.}

\item{formula}{A formula describing the model to be fitted}

\item{data}{A matrix or data frame containing variables in the model.}

\item{na.action}{Formula S3 method only: a function to specify the action to
be taken if NAs are found. The default action is for the procedure to fail.
An alternative is \code{na.omit}, which leads to rejection of cases with missing
values on any required variable. (NOTE: If given, this argument must be
named.)}
}
\value{
An object with S3 class "outercv"
\item{call}{the matched call}
\item{output}{Predictions on the left-out outer folds}
\item{outer_result}{List object of results from each outer fold containing
predictions on left-out outer folds, model result and number of filtered
predictors at each fold.}
\item{dimx}{vector of number of observations and number of predictors}
\item{outer_folds}{List of indices of outer test folds}
\item{final_fit}{Final fitted model on whole data}
\item{final_vars}{Column names of filtered predictors entering final model}
\item{summary_vars}{Summary statistics of filtered predictors}
\item{roc}{ROC AUC for binary classification where available.}
\item{summary}{Overall performance summary. Accuracy and balanced accuracy
for classification. ROC AUC for binary classification. RMSE for
regression.}
}
\description{
This is a convenience function designed to use a single loop of
cross-validation to quickly evaluate performance of specific models (random
forest, naive Bayes, lm, glm) with fixed hyperparameters and no tuning. If
tuning of parameters on data is required, full nested CV with inner CV is
needed to tune model hyperparameters (see \link{nestcv.train}).
}
\details{
Some predictive model functions do not have an x & y interface. If the
function specified by \code{model} requires a formula, \code{x} & \code{y} will be merged
into a dataframe with \code{model()} called with a formula equivalent to
\code{y ~ .}.

The S3 formula method for \code{outercv} is not really recommended with large
data sets - it is envisaged to be primarily used to compare
performance of more basic models e.g. \code{lm()} specified by formulae for
example incorporating interactions. NOTE: filtering is not available if
\code{outercv} is called with a formula - use the \code{x-y} interface instead.

An alternative method of tuning a single model with fixed parameters
is to use \link{nestcv.train} with \code{tuneGrid} set as a single row of a
data.frame. The parameters which are needed for a specific model can be
identified using \code{\link[caret:modelLookup]{caret::modelLookup()}}.

Case weights can be passed to model function which accept these, however
\code{outercv} assumes that these are passed to the model via an argument named
\code{weights}.

Note that in the case of \code{model = "lm"}, although additional arguments e.g.
\code{subset}, \code{weights}, \code{offset} are passed into the model function via
\code{"..."} the scoping is known to go awry. Avoid using these arguments with
\code{model = "lm"}.

\code{NA} handling differs between the default S3 method and the formula S3
method. The \code{na.option} argument takes a character string, while the more
typical \code{na.action} argument takes a function.
}
\examples{

## Classification example

## sigmoid function
sigmoid <- function(x) {1 / (1 + exp(-x))}

# load iris dataset and simulate a binary outcome
data(iris)
dt <- iris[, 1:4]
colnames(dt) <- c("marker1", "marker2", "marker3", "marker4")
dt <- as.data.frame(apply(dt, 2, scale))
x <- dt
y2 <- sigmoid(0.5 * dt$marker1 + 2 * dt$marker2) > runif(nrow(dt))
y2 <- factor(y2)

## Random forest
library(randomForest)
cvfit <- outercv(y2, x, "randomForest")
summary(cvfit)
plot(cvfit$roc)

## Mixture discriminant analysis (MDA)
if (requireNamespace("mda", quietly = TRUE)) {
  library(mda)
  cvfit <- outercv(y2, x, "mda", predict_type = "posterior")
  summary(cvfit)
}


## Example with continuous outcome
y <- -3 + 0.5 * dt$marker1 + 2 * dt$marker2 + rnorm(nrow(dt), 0, 2)
dt$outcome <- y

## simple linear model - formula interface
cvfit <- outercv(outcome ~ ., data = dt, model = "lm")
summary(cvfit)

## random forest for regression
cvfit <- outercv(y, x, "randomForest")
summary(cvfit)

## example with lm_filter() to reduce input predictors
cvfit <- outercv(y, x, "randomForest", filterFUN = lm_filter,
                 filter_options = list(nfilter = 2, p_cutoff = NULL))
summary(cvfit)

}
