% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mvgam.R
\name{mvgam}
\alias{mvgam}
\title{Fit a Bayesian dynamic GAM to a univariate or multivariate set of time series}
\usage{
mvgam(
  formula,
  trend_formula,
  knots,
  trend_knots,
  data,
  data_train,
  newdata,
  data_test,
  run_model = TRUE,
  prior_simulation = FALSE,
  return_model_data = FALSE,
  family = "poisson",
  share_obs_params = FALSE,
  use_lv = FALSE,
  n_lv,
  trend_map,
  trend_model = "None",
  drift = FALSE,
  chains = 4,
  burnin = 500,
  samples = 500,
  thin = 1,
  parallel = TRUE,
  threads = 1,
  priors,
  refit = FALSE,
  lfo = FALSE,
  residuals = TRUE,
  use_stan = TRUE,
  backend = getOption("brms.backend", "cmdstanr"),
  algorithm = getOption("brms.algorithm", "sampling"),
  autoformat = TRUE,
  save_all_pars = FALSE,
  max_treedepth,
  adapt_delta,
  jags_path,
  ...
)
}
\arguments{
\item{formula}{A \code{character} string specifying the GAM observation model formula. These are exactly like the formula
for a GLM except that smooth terms, \code{s()}, \code{te()}, \code{ti()}, \code{t2()}, as well as time-varying
\code{dynamic()} terms, can be added to the right hand side
to specify that the linear predictor depends on smooth functions of predictors
(or linear functionals of these). In \code{nmix()} family models, the \code{formula} is used to
set up a linear predictor for the detection probability. Details of the formula syntax used by \pkg{mvgam}
can be found in \code{\link{mvgam_formulae}}}

\item{trend_formula}{An optional \code{character} string specifying the GAM process model formula. If
supplied, a linear predictor will be modelled for the latent trends to capture process model evolution
separately from the observation model. Should not have a response variable specified on the left-hand side
of the formula (i.e. a valid option would be \code{~ season + s(year)}). Also note that you should not use
the identifier \code{series} in this formula to specify effects that vary across time series. Instead you should use
\code{trend}. This will ensure that models in which a \code{trend_map} is supplied will still work consistently
(i.e. by allowing effects to vary across process models, even when some time series share the same underlying
process model). This feature is only currently available for \code{RW()}, \code{AR()} and \code{VAR()} trend models.
In \code{nmix()} family models, the \code{trend_formula} is used to set up a linear predictor for the underlying
latent abundance}

\item{knots}{An optional \code{list} containing user specified knot values to be used for basis construction.
For most bases the user simply supplies the knots to be used, which must match up with the k value supplied
(note that the number of knots is not always just \code{k}). Different terms can use different numbers of knots,
unless they share a covariate}

\item{trend_knots}{As for \code{knots} above, this is an optional \code{list} of knot values for smooth
functions within the \code{trend_formula}}

\item{data}{A \code{dataframe} or \code{list} containing the model response variable and covariates
required by the GAM \code{formula} and optional \code{trend_formula}. Should include columns:
#'\itemize{
\item\code{series} (a \code{factor} index of the series IDs; the number of levels should be identical
to the number of unique series labels (i.e. \code{n_series = length(levels(data$series))}))
\item\code{time} (\code{numeric} or \code{integer} index of the time point for each observation).
For most dynamic trend types available in \code{mvgam} (see argument \code{trend_model}), time should be
measured in discrete, regularly spaced intervals (i.e. \code{c(1, 2, 3, ...)}). However you can
use irregularly spaced intervals if using \code{trend_model = CAR(1)}, though note that any
temporal intervals that are exactly \code{0} will be adjusted to a very small number
(\code{1e-12}) to prevent sampling errors. See an example of \code{CAR()} trends in \code{\link{CAR}}
}
Should also include any other variables to be included in the linear predictor of \code{formula}}

\item{data_train}{Deprecated. Still works in place of \code{data} but users are recommended to use
\code{data} instead for more seamless integration into \code{R} workflows}

\item{newdata}{Optional \code{dataframe} or \code{list} of test data containing at least \code{series} and \code{time}
in addition to any other variables included in the linear predictor of \code{formula}. If included, the
observations in variable \code{y} will be set to \code{NA} when fitting the model so that posterior
simulations can be obtained}

\item{data_test}{Deprecated. Still works in place of \code{newdata} but users are recommended to use
\code{newdata} instead for more seamless integration into \code{R} workflows}

\item{run_model}{\code{logical}. If \code{FALSE}, the model is not fitted but instead the function will
return the model file and the data / initial values that are needed to fit the model outside of \code{mvgam}}

\item{prior_simulation}{\code{logical}. If \code{TRUE}, no observations are fed to the model, and instead
simulations from prior distributions are returned}

\item{return_model_data}{\code{logical}. If \code{TRUE}, the list of data that is needed to fit the
model is returned, along with the initial values for smooth and AR parameters, once the model is fitted.
This will be helpful if users wish to modify the model file to add
other stochastic elements that are not currently available in \code{mvgam}.
Default is \code{FALSE} to reduce
the size of the returned object, unless \code{run_model == FALSE}}

\item{family}{\code{family} specifying the exponential observation family for the series. Currently supported
families are:
\itemize{
\item\code{gaussian()} for real-valued data
\item\code{betar()} for proportional data on \verb{(0,1)}
\item\code{lognormal()} for non-negative real-valued data
\item\code{student_t()} for real-valued data
\item\code{Gamma()} for non-negative real-valued data
\item\code{bernoulli()} for binary data
\item\code{nb()} for count data
\item\code{poisson()} for count data
\item\code{binomial()} for count data with imperfect detection when the number of trials is known;
note that the \code{cbind()} function must be used to bind the discrete observations and the number
of trials
\item\code{nmix()} for count data with imperfect detection when the number of trials
is unknown and should be modeled via a State-Space N-Mixture model.
The latent states are Poisson, capturing the 'true' latent
abundance, while the observation process is Binomial to account for
imperfect detection.
See \code{\link{mvgam_families}} for an example of how to use this family}
Note that only \code{nb()} and \code{poisson()} are available if using \code{JAGS} as the backend.
Default is \code{poisson()}.
See \code{\link{mvgam_families}} for more details}

\item{share_obs_params}{\code{logical}. If \code{TRUE} and the \code{family}
has additional family-specific observation parameters (e.g. variance components in
\code{student_t()} or \code{gaussian()}, or dispersion parameters in \code{nb()} or \code{betar()}),
these parameters will be shared across all series. This is handy if you have multiple
time series that you believe share some properties, such as being from the same
species over different spatial units. Default is \code{FALSE}.}

\item{use_lv}{\code{logical}. If \code{TRUE}, use dynamic factors to estimate series'
latent trends in a reduced dimension format. Only available for
\code{RW()}, \code{AR()} and \code{GP()} trend models. Defaults to \code{FALSE}}

\item{n_lv}{\code{integer} the number of latent dynamic factors to use if \code{use_lv == TRUE}.
Cannot be \code{> n_series}. Defaults arbitrarily to \code{min(2, floor(n_series / 2))}}

\item{trend_map}{Optional \code{data.frame} specifying which series should depend on which latent
trends. Useful for allowing multiple series to depend on the same latent trend process, but with
different observation processes. If supplied, a latent factor model is set up by setting
\code{use_lv = TRUE} and using the mapping to set up the shared trends. Needs to have column names
\code{series} and \code{trend}, with integer values in the \code{trend} column to state which trend each series
should depend on. The \code{series} column should have a single unique entry for each series in the
data (names should perfectly match factor levels of the \code{series} variable in \code{data}). See examples
for details}

\item{trend_model}{\code{character} or  \code{function} specifying the time series dynamics for the latent trend. Options are:
\itemize{
\item \code{None} (no latent trend component; i.e. the GAM component is all that contributes to the linear predictor,
and the observation process is the only source of error; similarly to what is estimated by \code{\link[mgcv]{gam}})
\item \code{'RW'} or \code{RW()}
\item \code{'AR1'} or \code{AR(p = 1)}
\item \code{'AR2'} or \code{AR(p = 2)}
\item \code{'AR3'} or \code{AR(p = 3)}
\item \code{'CAR1'} or \code{CAR(p = 1)}
\item \code{'VAR1'}  or \code{VAR()}(only available in \code{Stan})
\item \verb{'PWlogistic}, \code{'PWlinear'} or \code{PW()} (only available in \code{Stan})
\item \code{'GP'} or \code{GP()} (Gaussian Process with squared exponential kernel;
only available in \code{Stan})}

For all trend types apart from \code{GP()}, \code{CAR()} and \code{PW()}, moving average and/or correlated
process error terms can also be estimated (for example, \code{RW(cor = TRUE)} will set up a
multivariate Random Walk if \code{n_series > 1}). See \link{mvgam_trends} for more details}

\item{drift}{\code{logical} estimate a drift parameter in the latent trend components. Useful if the latent
trend is expected to broadly follow a non-zero slope. Only available for
\code{RW()} and \code{AR()} trend models. Note that if the latent trend is more or less stationary,
the drift parameter can become unidentifiable, especially if an intercept term is included in the GAM linear
predictor (which it is by default when calling \code{\link[mgcv]{jagam}}). Drift parameters will also likely
be unidentifiable if using dynamic factor models. Therefore this defaults to \code{FALSE}}

\item{chains}{\code{integer} specifying the number of parallel chains for the model. Ignored
if \code{algorithm \%in\% c('meanfield', 'fullrank', 'pathfinder', 'laplace')}}

\item{burnin}{\code{integer} specifying the number of warmup iterations of the Markov chain to run
to tune sampling algorithms. Ignored
if \code{algorithm \%in\% c('meanfield', 'fullrank', 'pathfinder', 'laplace')}}

\item{samples}{\code{integer} specifying the number of post-warmup iterations of the Markov chain to run for
sampling the posterior distribution}

\item{thin}{Thinning interval for monitors.  Ignored
if \code{algorithm \%in\% c('meanfield', 'fullrank', 'pathfinder', 'laplace')}}

\item{parallel}{\code{logical} specifying whether multiple cores should be used for
generating MCMC simulations in parallel. If \code{TRUE}, the number of cores to use will be
\code{min(c(chains, parallel::detectCores() - 1))}}

\item{threads}{\code{integer} Experimental option to use multithreading for within-chain
parallelisation in \code{Stan}. We recommend its use only if you are experienced with
\code{Stan}'s \code{reduce_sum} function and have a slow running model that cannot be sped
up by any other means. Only available for some families(\code{poisson()}, \code{nb()}, \code{gaussian()}) and
when using \code{Cmdstan} as the backend}

\item{priors}{An optional \code{data.frame} with prior
definitions (in JAGS or Stan syntax). if using Stan, this can also be an object of
class \code{brmsprior} (see. \code{\link[brms]{prior}} for details). See \link{get_mvgam_priors} and
'Details' for more information on changing default prior distributions}

\item{refit}{Logical indicating whether this is a refit, called using \link{update.mvgam}. Users should leave
as \code{FALSE}}

\item{lfo}{Logical indicating whether this is part of a call to \link{lfo_cv.mvgam}. Returns a
lighter version of the model with no residuals and fewer monitored parameters to speed up
post-processing. But other downstream functions will not work properly, so users should always
leave this set as \code{FALSE}}

\item{residuals}{Logical indicating whether to compute series-level randomized quantile residuals and include
them as part of the returned object. Defaults to \code{TRUE}, but you can set to \code{FALSE} to save
computational time and reduce the size of the returned object (users can always add residuals to
an object of class \code{mvgam} using \link{add_residuals})}

\item{use_stan}{Logical. If \code{TRUE}, the model will be compiled and sampled using
Hamiltonian Monte Carlo with a call to \code{\link[cmdstanr]{cmdstan_model}} or
a call to \code{\link[rstan]{stan}}. Note that
there are many more options when using \code{Stan} vs \code{JAGS}}

\item{backend}{Character string naming the package to use as the backend for fitting
the Stan model (if \code{use_stan = TRUE}). Options are "cmdstanr" (the default) or "rstan". Can be set globally
for the current R session via the \code{"brms.backend"} option (see \code{\link{options}}). Details on
the rstan and cmdstanr packages are available at https://mc-stan.org/rstan/ and
https://mc-stan.org/cmdstanr/, respectively}

\item{algorithm}{Character string naming the estimation approach to use.
Options are \code{"sampling"} for MCMC (the default), \code{"meanfield"} for
variational inference with factorized normal distributions,
\code{"fullrank"} for variational inference with a multivariate normal
distribution, \code{"laplace"} for a Laplace approximation (only available
when using cmdstanr as the backend) or \code{"pathfinder"} for the pathfinder
algorithm (only currently available when using cmdstanr as the backend).
Can be set globally for the current \R session via the
\code{"brms.algorithm"} option (see \code{\link{options}}). Limited testing
suggests that \code{"meanfield"} performs best out of the non-MCMC approximations for
dynamic GAMs, possibly because of the difficulties estimating covariances among the
many spline parameters and latent trend parameters. But rigorous testing has not
been carried out}

\item{autoformat}{\code{Logical}. Use the \code{stanc} parser to automatically format the
\code{Stan} code and check for deprecations. Defaults to \code{TRUE}}

\item{save_all_pars}{\code{Logical} flag to indicate if draws from all
variables defined in Stan's \code{parameters} block should be saved
(default is \code{FALSE}).}

\item{max_treedepth}{positive integer placing a cap on the number of simulation steps evaluated during each iteration when
\code{use_stan == TRUE}. Default is \code{12}. Increasing this value can sometimes help with exploration of complex
posterior geometries, but it is rarely fruitful to go above a \code{max_treedepth} of \code{14}}

\item{adapt_delta}{positive numeric between \code{0} and \code{1} defining the target average proposal acceptance probability
during Stan's adaptation period, if \code{use_stan == TRUE}. Default is \code{0.8}. In general you should not need to change adapt_delta
unless you see a warning message about divergent transitions, in which case you can increase adapt_delta from the default
to a value closer to \code{1} (e.g. from \code{0.95} to \code{0.99}, or from \code{0.99} to \code{0.999}, etc).
The step size used by the numerical integrator is a function of \code{adapt_delta} in that increasing
\code{adapt_delta} will result in a smaller step size and fewer divergences. Increasing \code{adapt_delta} will
typically result in a slower sampler, but it will always lead to a more robust sampler}

\item{jags_path}{Optional character vector specifying the path to the location of the \code{JAGS} executable (.exe) to use
for modelling if \code{use_stan == FALSE}. If missing, the path will be recovered from a call to \code{\link[runjags]{findjags}}}

\item{...}{Further arguments passed to Stan.
For \code{backend = "rstan"} the arguments are passed to
\code{\link[rstan]{sampling}} or \code{\link[rstan]{vb}}.
For \code{backend = "cmdstanr"} the arguments are passed to the
\code{cmdstanr::sample}, \code{cmdstanr::variational},
\code{cmdstanr::laplace} or
\code{cmdstanr::pathfinder} method}
}
\value{
A \code{list} object of class \code{mvgam} containing model output, the text representation of the model file,
the mgcv model output (for easily generating simulations at
unsampled covariate values), Dunn-Smyth residuals for each series and key information needed
for other functions in the package. See \code{\link{mvgam-class}} for details.
Use \code{methods(class = "mvgam")} for an overview on available methods.
}
\description{
This function estimates the posterior distribution for Generalised Additive Models (GAMs) that can include
smooth spline functions, specified in the GAM formula, as well as latent temporal processes,
specified by \code{trend_model}. Further modelling options include State-Space representations to allow covariates
and dynamic processes to occur on the latent 'State' level while also capturing observation-level effects.
Prior specifications are flexible and explicitly encourage users to apply
prior distributions that actually reflect their beliefs. In addition, model fits can easily be assessed and
compared with posterior predictive checks, forecast comparisons and leave-one-out / leave-future-out cross-validation.
}
\details{
Dynamic GAMs are useful when we wish to predict future values from time series that show temporal dependence
but we do not want to rely on extrapolating from a smooth term (which can sometimes lead to unpredictable and unrealistic behaviours).
In addition, smooths can often try to wiggle excessively to capture any autocorrelation that is present in a time series,
which exacerbates the problem of forecasting ahead. As GAMs are very naturally viewed through a Bayesian lens, and we often
must model time series that show complex distributional features and missing data, parameters for \code{mvgam} models are estimated
in a Bayesian framework using Markov Chain Monte Carlo by default. A general overview is provided
in the primary vignettes: \code{vignette("mvgam_overview")} and \code{vignette("data_in_mvgam")}.
For a full list of available vignettes see \code{vignette(package = "mvgam")}
\cr
\cr
\emph{Formula syntax}: Details of the formula syntax used by \pkg{mvgam} can be found in
\code{\link{mvgam_formulae}}. Note that it is possible to supply an empty formula where
there are no predictors or intercepts in the observation model (i.e. \code{y ~ 0} or \code{y ~ -1}).
In this case, an intercept-only observation model will be set up but the intercept coefficient
will be fixed at zero. This can be handy if you wish to fit pure State-Space models where
the variation in the dynamic trend controls the average expectation, and/or where intercepts
are non-identifiable (as in piecewise trends, see examples below)
\cr
\cr
\emph{Families and link functions}: Details of families supported by \pkg{mvgam} can be found in
\code{\link{mvgam_families}}.
\cr
\cr
\emph{Trend models}: Details of latent trend dynamic models supported by \pkg{mvgam} can be found in
\code{\link{mvgam_trends}}.
\cr
\cr
\emph{Priors}: A \code{\link[mgcv]{jagam}} model file is generated from \code{formula} and
modified to include any latent
temporal processes. Default priors for intercepts and any scale parameters are generated
using the same practice as \pkg{brms}. Prior distributions for most important model parameters can be
altered by the user to inspect model
sensitivities to given priors (see \code{\link{get_mvgam_priors}} for details).
Note that latent trends are estimated on the link scale so choose priors
accordingly. However more control over the model specification can be accomplished by first using \code{mvgam} as a
baseline, then editing the returned model accordingly. The model file can be edited and run outside
of \code{mvgam} by setting \code{run_model = FALSE} and this is encouraged for complex
modelling tasks. Note, no priors are
formally checked to ensure they are in the right syntax for the respective
probabilistic modelling framework, so it is
up to the user to ensure these are correct (i.e. use \code{dnorm} for normal
densities in \code{JAGS}, with the mean and precision
parameterisation; but use \code{normal} for normal densities in \code{Stan}, with the mean
and standard deviation parameterisation)
\cr
\cr
\emph{Random effects}: For any smooth terms using the random effect basis (\code{\link[mgcv]{smooth.construct.re.smooth.spec}}),
a non-centred parameterisation is automatically employed to avoid degeneracies that are common in hierarchical models.
Note however that centred versions may perform better for series that are particularly informative, so as with any
foray into Bayesian modelling, it is worth building an understanding of the model's assumptions and limitations by following a
principled workflow. Also note that models are parameterised using \code{drop.unused.levels = FALSE} in \code{\link[mgcv]{jagam}}
to ensure predictions can be made for all levels of the supplied factor variable
\cr
\cr
\emph{Observation level parameters}: When more than one series is included in \code{data} and an
observation family that contains more than one parameter is used, additional observation family parameters
(i.e. \code{phi} for \code{nb()} or \code{sigma} for \code{gaussian()}) are
by default estimated independently for each series. But if you wish for the series to share
the same observation parameters, set \code{share_obs_params = TRUE}
\cr
\cr
\emph{Factor regularisation}: When using a dynamic factor model for the trends with \code{JAGS} factor precisions are given
regularized penalty priors to theoretically allow some factors to be dropped from the model by squeezing increasing
factors' variances to zero. This is done to help protect against selecting too many latent factors than are needed to
capture dependencies in the data, so it can often be advantageous to set \code{n_lv} to a slightly larger number. However
larger numbers of factors do come with additional computational costs so these should be balanced as well. When using
\code{Stan}, all factors are parameterised with fixed variance parameters
\cr
\cr
\emph{Residuals}: For each series, randomized quantile (i.e. Dunn-Smyth) residuals are calculated for inspecting model diagnostics
If the fitted model is appropriate then Dunn-Smyth residuals will be standard normal in distribution and no
autocorrelation will be evident. When a particular observation is missing, the residual is calculated by comparing independent
draws from the model's posterior distribution
\cr
\cr
\emph{Using Stan}: \code{mvgam} is primarily designed to use Hamiltonian Monte Carlo for parameter estimation
via the software \code{Stan} (using either the \code{cmdstanr} or \code{rstan} interface).
There are great advantages when using \code{Stan} over Gibbs / Metropolis Hastings samplers, which includes the option
to estimate smooth latent trends via \href{https://arxiv.org/abs/2004.11408}{Hilbert space approximate Gaussian Processes}.
This often makes sense for ecological series, which we expect to change smoothly. In \code{mvgam}, latent squared
exponential GP trends are approximated using by default \code{20} basis functions, which saves computational
costs compared to fitting full GPs while adequately estimating
GP \code{alpha} and \code{rho} parameters. Because of the many advantages of \code{Stan} over \code{JAGS},
\emph{further development of the package will only be applied to \code{Stan}}. This includes the planned addition
of more response distributions, plans to handle zero-inflation, and plans to incorporate a greater
variety of trend models. Users are strongly encouraged to opt for \code{Stan} over \code{JAGS} in any proceeding workflows
}
\examples{
\donttest{
# Simulate a collection of three time series that have shared seasonal dynamics
dat <- sim_mvgam(T = 80, n_series = 3, prop_missing = 0.1,
                prop_trend = 0.6)

# Plot key summary statistics for a single series
plot_mvgam_series(data = dat$data_train, series = 1)

# Plot all series together
plot_mvgam_series(data = dat$data_train, series = 'all')

# Formulate a model using Stan where series share a cyclic smooth for
# seasonality and each series has an independent random walk temporal process;
# Set run_model = FALSE to inspect the returned objects
mod1 <- mvgam(formula = y ~ s(season, bs = 'cc', k = 6),
             data = dat$data_train,
             trend_model = RW(),
             family = poisson(),
             use_stan = TRUE,
             run_model = FALSE)

# View the model code in Stan language
code(mod1)

# Now fit the model
mod1 <- mvgam(formula = y ~ s(season, bs = 'cc', k = 6),
              data = dat$data_train,
              trend_model = RW(),
              family = poisson(),
              chains = 2)

# Extract the model summary
summary(mod1)

# Plot the estimated historical trend and forecast for one series
plot(mod1, type = 'trend', series = 1)
plot(mod1, type = 'forecast', series = 1)

# Residual diagnostics
plot(mod1, type = 'residuals', series = 1)
resids <- residuals(mod1)
str(resids)

# Compute the forecast using covariate information in data_test
fc <- forecast(mod1, newdata = dat$data_test)
str(fc)
plot(fc)

# Plot the estimated seasonal smooth function
plot(mod1, type = 'smooths')

# Plot estimated first derivatives of the smooth
plot(mod1, type = 'smooths', derivatives = TRUE)

# Plot partial residuals of the smooth
plot(mod1, type = 'smooths', residuals = TRUE)

# Plot posterior realisations for the smooth
plot(mod1, type = 'smooths', realisations = TRUE)

# Plot conditional response predictions using marginaleffects
conditional_effects(mod1)
plot_predictions(mod1, condition = 'season', points = 0.5)

# Generate posterior predictive checks using bayesplot
pp_check(mod1)

# Extract observation model beta coefficient draws as a data.frame
beta_draws_df <- as.data.frame(mod1, variable = 'betas')
head(beta_draws_df)
str(beta_draws_df)

# Investigate model fit
mc.cores.def <- getOption('mc.cores')
options(mc.cores = 1)
loo(mod1)
options(mc.cores = mc.cores.def)


# Example of supplying a trend_map so that some series can share
# latent trend processes
sim <- sim_mvgam(n_series = 3)
mod_data <- sim$data_train

# Here, we specify only two latent trends; series 1 and 2 share a trend,
# while series 3 has it's own unique latent trend
trend_map <- data.frame(series = unique(mod_data$series),
                       trend = c(1, 1, 2))

# Fit the model using AR1 trends
mod <- mvgam(y ~ s(season, bs = 'cc', k = 6),
             trend_map = trend_map,
             trend_model = AR(),
             data = mod_data,
             return_model_data = TRUE,
             chains = 2)

# The mapping matrix is now supplied as data to the model in the 'Z' element
mod1$model_data$Z
code(mod)

# The first two series share an identical latent trend; the third is different
plot(mod, type = 'trend', series = 1)
plot(mod, type = 'trend', series = 2)
plot(mod, type = 'trend', series = 3)


# Example of how to use dynamic coefficients
# Simulate a time-varying coefficient for the effect of temperature
set.seed(123)
N <- 200
beta_temp <- vector(length = N)
beta_temp[1] <- 0.4
for(i in 2:N){
 beta_temp[i] <- rnorm(1, mean = beta_temp[i - 1] - 0.0025, sd = 0.05)
}
plot(beta_temp)

# Simulate a covariate called 'temp'
temp <- rnorm(N, sd = 1)

# Simulate the Gaussian observation process
out <- rnorm(N, mean = 4 + beta_temp * temp,
             sd = 0.5)

# Gather necessary data into a data.frame; split into training / testing
data = data.frame(out, temp, time = seq_along(temp))
data_train <- data[1:180,]
data_test <- data[181:200,]

# Fit the model using the dynamic() formula helper
mod <- mvgam(out ~
              dynamic(temp,
                      scale = FALSE,
                      k = 40),
             family = gaussian(),
             data = data_train,
             newdata = data_test,
             chains = 2)

# Inspect the model summary, forecast and time-varying coefficient distribution
summary(mod)
plot(mod, type = 'smooths')
fc <- forecast(mod, newdata = data_test)
plot(fc)

# Propagating the smooth term shows how the coefficient is expected to evolve
plot_mvgam_smooth(mod, smooth = 1, newdata = data)
abline(v = 180, lty = 'dashed', lwd = 2)
points(beta_temp, pch = 16)


# Example showing how to incorporate an offset; simulate some count data
# with different means per series
set.seed(100)
dat <- sim_mvgam(prop_trend = 0, mu = c(0, 2, 2), seasonality = 'hierarchical')

# Add offset terms to the training and testing data
dat$data_train$offset <- 0.5 * as.numeric(dat$data_train$series)
dat$data_test$offset <- 0.5 * as.numeric(dat$data_test$series)

# Fit a model that includes the offset in the linear predictor as well as
# hierarchical seasonal smooths
mod <- mvgam(formula = y ~ offset(offset) +
              s(series, bs = 're') +
              s(season, bs = 'cc') +
              s(season, by = series, m = 1, k = 5),
             data = dat$data_train,
             chains = 2)

# Inspect the model file to see the modification to the linear predictor
# (eta)
code(mod)

# Forecasts for the first two series will differ in magnitude
fc <- forecast(mod, newdata = dat$data_test)
layout(matrix(1:2, ncol = 2))
plot(fc, series = 1, ylim = c(0, 75))
plot(fc, series = 2, ylim = c(0, 75))
layout(1)

# Changing the offset for the testing data should lead to changes in
# the forecast
dat$data_test$offset <- dat$data_test$offset - 2
fc <- forecast(mod, newdata = dat$data_test)
plot(fc)

# Relative Risks can be computed by fixing the offset to the same value
# for each series
dat$data_test$offset <- rep(1, NROW(dat$data_test))
preds_rr <- predict(mod, type = 'link', newdata = dat$data_test,
                    summary = FALSE)
series1_inds <- which(dat$data_test$series == 'series_1')
series2_inds <- which(dat$data_test$series == 'series_2')

# Relative Risks are now more comparable among series
layout(matrix(1:2, ncol = 2))
plot(preds_rr[1, series1_inds], type = 'l', col = 'grey75',
     ylim = range(preds_rr),
     ylab = 'Series1 Relative Risk', xlab = 'Time')
for(i in 2:50){
 lines(preds_rr[i, series1_inds], col = 'grey75')
}

plot(preds_rr[1, series2_inds], type = 'l', col = 'darkred',
     ylim = range(preds_rr),
     ylab = 'Series2 Relative Risk', xlab = 'Time')
for(i in 2:50){
 lines(preds_rr[i, series2_inds], col = 'darkred')
 }
layout(1)


# Example showcasing how cbind() is needed for Binomial observations
# Simulate two time series of Binomial trials
trials <- sample(c(20:25), 50, replace = TRUE)
x <- rnorm(50)
detprob1 <- plogis(-0.5 + 0.9*x)
detprob2 <- plogis(-0.1 -0.7*x)
dat <- rbind(data.frame(y = rbinom(n = 50, size = trials, prob = detprob1),
                        time = 1:50,
                        series = 'series1',
                        x = x,
                        ntrials = trials),
             data.frame(y = rbinom(n = 50, size = trials, prob = detprob2),
                        time = 1:50,
                        series = 'series2',
                        x = x,
                        ntrials = trials))
dat <- dplyr::mutate(dat, series = as.factor(series))
dat <- dplyr::arrange(dat, time, series)
plot_mvgam_series(data = dat, series = 'all')

# Fit a model using the binomial() family; must specify observations
# and number of trials in the cbind() wrapper
mod <- mvgam(cbind(y, ntrials) ~ series + s(x, by = series),
             family = binomial(),
             data = dat,
             chains = 2)
summary(mod)
pp_check(mod, type = "bars_grouped",
         group = "series", ndraws = 50)
pp_check(mod, type = "ecdf_overlay_grouped",
         group = "series", ndraws = 50)
conditional_effects(mod, type = 'link')
}
}
\references{
Nicholas J Clark & Konstans Wells (2020). Dynamic generalised additive models (DGAMs) for forecasting discrete ecological time series.
Methods in Ecology and Evolution. 14:3, 771-784.
}
\seealso{
\code{\link[mgcv]{jagam}}, \code{\link[mgcv]{gam}}, \code{\link[mgcv]{gam.models}},
}
\author{
Nicholas J Clark
}
