% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/assertions.R
\name{mt_is_track_id_cleaved}
\alias{mt_is_track_id_cleaved}
\alias{mt_is_time_ordered}
\alias{mt_has_unique_location_time_records}
\alias{mt_is_time_ordered_non_empty_points}
\alias{mt_has_no_empty_points}
\alias{mt_is_move2}
\title{Functions for asserting properties of a \code{move2} object}
\usage{
mt_is_track_id_cleaved(x)

mt_is_time_ordered(x, non_zero = FALSE)

mt_has_unique_location_time_records(x)

mt_is_time_ordered_non_empty_points(x, non_zero = FALSE)

mt_has_no_empty_points(x)

mt_is_move2(x)
}
\arguments{
\item{x}{a \code{move2} object}

\item{non_zero}{If \code{TRUE} only intervals longer than 0 are considered ordered (i.e. no coinciding timestamps),
if \code{FALSE} also 0 intervals are considered ordered}
}
\value{
a logical value if the asserted property is \code{TRUE} or \code{FALSE}
}
\description{
\itemize{
\item \code{mt_is_track_id_cleaved()} asserts all tracks are grouped in the data set, they occur consecutively.
\item \code{mt_is_time_ordered()} checks if all tracks are groups and if timestamps within a track are ascending (i.e. the
time differences between successive locations are equal or above 0).
\item \code{mt_has_unique_location_time_records()} checks if all records with a location have a unique timestamp (i.e. checks
for duplicated timestamps within a track).
\item \code{mt_is_time_ordered_non_empty_points()} this assertion combines the \code{mt_is_time_ordered()} and
\code{mt_has_no_empty_points()} assertions and thus ensures that each record has a location and timestamps are ordered.
\item \code{mt_has_no_empty_points()} asserts all geometries are points and that there are no empty points.
\item \code{mt_is_move2()} asserts \code{x} inherits the class \code{move2}
}
}
\details{
For these functions an \code{\link[assertthat]{on_failure}} error
function is defined. This results in meaningful error messages when the
function is used in combination with \code{\link[assertthat]{assert_that}}. These functions can also be used in
normal logical operations as \code{TRUE} or \code{FALSE} is returned.
}
\examples{
## examples of what to do if assertion if FALSE
n <- 8
data <- data.frame(x = cumsum(rnorm(n)), y = cumsum(rnorm(n)),
                   time = seq(n), track = sample(c("a","b"), size=n, replace=TRUE))
data <- rbind(data, data[sample(nrow(data),2),]) # adding duplicate timestamps
mv <- mt_as_move2(data, coords = c("x", "y"),
                  time_column = "time",
                  track_id_column = "track")
mv$geometry[c(1, 3)] <- sf::st_point() # adding empty locations

mt_is_track_id_cleaved(mv)
mv <- dplyr::arrange(mv, mt_track_id(mv))

mt_is_time_ordered(mv)
mv <- dplyr::arrange(mv, mt_track_id(mv), mt_time(mv))

mt_has_unique_location_time_records(mv)
mv <- mt_filter_unique(mv)

mt_has_no_empty_points(mv)
mv <- dplyr::filter(mv, !sf::st_is_empty(mv))

## example of using the assertions with assertthat
if (requireNamespace("assertthat")) {
  m <- mt_sim_brownian_motion(t = 1:2, tracks = 2)
  assertthat::see_if(mt_is_track_id_cleaved(m))
  assertthat::see_if(mt_is_track_id_cleaved(m[c(3, 1, 2, 4), ]))
  assertthat::see_if(mt_is_time_ordered(m[c(2:1, 3, 4), ]))
  assertthat::see_if(mt_has_unique_location_time_records(m[c(1, 1, 2, 3, 4), ]))
  assertthat::see_if(mt_is_move2(m$time))
}

}
