% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tuneParams.R
\name{tuneParams}
\alias{tuneParams}
\title{Hyperparameter tuning.}
\usage{
tuneParams(learner, task, resampling, measures, par.set, control,
  show.info = getMlrOption("show.info"))
}
\arguments{
\item{learner}{[\code{\link{Learner}} | \code{character(1)}]\cr
The learner.
If you pass a string the learner will be created via \code{\link{makeLearner}}.}

\item{task}{[\code{\link{Task}}]\cr
The task.}

\item{resampling}{[\code{\link{ResampleInstance}} | \code{\link{ResampleDesc}}]\cr
Resampling strategy to evaluate points in hyperparameter space. If you pass a description,
it is instantiated once at the beginning by default, so all points are
evaluated on the same training/test sets.
If you want to change that behavior, look at \code{\link{TuneControl}}.}

\item{measures}{[list of \code{\link{Measure}} | \code{\link{Measure}}]\cr
Performance measures to evaluate. The first measure, aggregated by the first aggregation function
is optimized, others are simply evaluated.
Default is the default measure for the task, see here \code{\link{getDefaultMeasure}}.}

\item{par.set}{[\code{\link[ParamHelpers]{ParamSet}}]\cr
Collection of parameters and their constraints for optimization.
Dependent parameters with a \code{requires} field must use \code{quote} and not
\code{expression} to define it.}

\item{control}{[\code{\link{TuneControl}}]\cr
Control object for search method. Also selects the optimization algorithm for tuning.}

\item{show.info}{[\code{logical(1)}]\cr
Print verbose output on console?
Default is set via \code{\link{configureMlr}}.}
}
\value{
[\code{\link{TuneResult}}].
}
\description{
Optimizes the hyperparameters of a learner.
Allows for different optimization methods, such as grid search, evolutionary strategies,
iterated F-race, etc. You can select such an algorithm (and its settings)
by passing a corresponding control object. For a complete list of implemented algorithms look at
\code{\link{TuneControl}}.

Multi-criteria tuning can be done with \code{\link{tuneParamsMultiCrit}}.
}
\note{
If you would like to include results from the training data set, make
sure to appropriately adjust the resampling strategy and the aggregation for
the measure. See example code below.
}
\examples{
# a grid search for an SVM (with a tiny number of points...)
# note how easily we can optimize on a log-scale
ps = makeParamSet(
  makeNumericParam("C", lower = -12, upper = 12, trafo = function(x) 2^x),
  makeNumericParam("sigma", lower = -12, upper = 12, trafo = function(x) 2^x)
)
ctrl = makeTuneControlGrid(resolution = 2L)
rdesc = makeResampleDesc("CV", iters = 2L)
res = tuneParams("classif.ksvm", iris.task, rdesc, par.set = ps, control = ctrl)
print(res)
# access data for all evaluated points
print(head(as.data.frame(res$opt.path)))
print(head(as.data.frame(res$opt.path, trafo = TRUE)))
# access data for all evaluated points - alternative
print(head(generateHyperParsEffectData(res)))
print(head(generateHyperParsEffectData(res, trafo = TRUE)))

\dontrun{
# we optimize the SVM over 3 kernels simultanously
# note how we use dependent params (requires = ...) and iterated F-racing here
ps = makeParamSet(
  makeNumericParam("C", lower = -12, upper = 12, trafo = function(x) 2^x),
  makeDiscreteParam("kernel", values = c("vanilladot", "polydot", "rbfdot")),
  makeNumericParam("sigma", lower = -12, upper = 12, trafo = function(x) 2^x,
    requires = quote(kernel == "rbfdot")),
  makeIntegerParam("degree", lower = 2L, upper = 5L,
    requires = quote(kernel == "polydot"))
)
print(ps)
ctrl = makeTuneControlIrace(maxExperiments = 5, nbIterations = 1, minNbSurvival = 1)
rdesc = makeResampleDesc("Holdout")
res = tuneParams("classif.ksvm", iris.task, rdesc, par.set = ps, control = ctrl)
print(res)
print(head(as.data.frame(res$opt.path)))

# include the training set performance as well
rdesc = makeResampleDesc("Holdout", predict = "both")
res = tuneParams("classif.ksvm", iris.task, rdesc, par.set = ps,
  control = ctrl, measures = list(mmce, setAggregation(mmce, train.mean)))
print(res)
print(head(as.data.frame(res$opt.path)))
}
}
\seealso{
\code{\link{generateHyperParsEffectData}}

Other tune: \code{\link{TuneControl}},
  \code{\link{getNestedTuneResultsOptPathDf}},
  \code{\link{getNestedTuneResultsX}},
  \code{\link{getTuneResult}},
  \code{\link{makeModelMultiplexerParamSet}},
  \code{\link{makeModelMultiplexer}},
  \code{\link{makeTuneWrapper}},
  \code{\link{tuneThreshold}}
}
