% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/mixedMemModel.R
\name{mixedMemModel}
\alias{mixedMemModel}
\title{Constructor for a Mixed Membership Model Object}
\usage{
mixedMemModel(Total, J, Rj, Nijr, K, Vj, alpha, theta, phi = NULL,
  delta = NULL, dist, obs, fixedObs = NULL, P = NULL, beta = NULL)
}
\arguments{
\item{Total}{the number of individuals in the sample.}

\item{J}{the number of variables observed on each individual.}

\item{Rj}{a vector of length J specifying the number of repeated measurements
for each variable.}

\item{Nijr}{an array with dimension (\code{Total}, \code{J}, \code{max(Rj)}) indicating the number
of ranking levels for each variable and each replication. For multinomial and Bernoulli
variables, \code{Nijr}[i,j,r] = 1. For rank variables, \code{Nijr}[i,j,r] indicates the
number of alternatives ranked.}

\item{K}{the number of sub-populations.}

\item{Vj}{a vector of length \code{J} specifying the number of possible candidates
for each variable. For a Bernoulli variable \code{Vj}[j] = 1.}

\item{alpha}{a vector of length \code{K} which is the parameter for Dirichlet
membership distribution.}

\item{theta}{an array with dimensions (\code{J},\code{K},\code{max(Vj)}) which governs the variable
distributions. The parameter \code{theta}[j,k,] governs the distribution of variable J for a complete member of sub-population k. For instance, if variable j is a Bernoulli variable, theta[j,k,1] is the probability of success; if
variable j is a multinomial variable, \code{theta}[j,k, 1:Vj[j]] is the probability for each of the \code{Vj}[j] categories; if variable j
is a rank variable, \code{theta}[j,k, 1:Vj[j]] are the support parameters for each of the \code{Vj}[j] alternatives. Since the dimension of the relevant parameters
can differ across variables, any unused elements of the array should be set to 0, while all other elements should be positive.}

\item{phi}{an array with dimensions (\code{Total},\code{K}) which specifies the variational
parameters for the membership vectors, \eqn{\lambda}. The default group membership initialization is uniform across all groups (phi[i,k] = 1/K for all k).
The default initialization is highly recommended.}

\item{delta}{an array with dimensions (\code{Total},\code{J},\code{max(Rj)}, \code{max(Nijr)}, \code{K}) which specifies
the variational parameters for the context vectors Z. The default initialization is
 uniform across all sub-populations (\code{delta}[i, j, r, n, k] = 1/K for all k).}

\item{dist}{a vector of length \code{J} specifying variable types. Options are
"bernoulli", "multinomial" or "rank" corresponing to the distributions
 of the observed variables.}

\item{obs}{an array with dimensions (\code{Total},\code{J},\code{max(Rj)}, \code{max(Nijr)}) corresponding to
the observed data. For Bernoulli random variables, the data consist of 0/1's.
For multinomial or rank data the data consist of integers 0,1,\ldots,(Vj[j]-1).}

\item{fixedObs}{an array with dimensions (1, \code{J}, \code{max(Rj)}, \code{max(Nijr)}) corresonding to the observed
responses for a fixed group in the extended GoM model from Erosheva et al (2007)}

\item{P}{scalar between 0 and 1 corresponding to initial value for the proportion of individuals in the fixed group}

\item{beta}{scalar between 0 and 1 corresponding to the initial value of beta, the conditional probability of being in the fixed group
for an individual who's observed responses match the fixed group.}
}
\value{
returns an object of class \code{mixedMemModel}
}
\description{
Constructor for a \code{mixedMemModel} object which can be used for analysis
in the \code{mixedMem} package.
}
\details{
The function returns an object of \code{mixedMemModel} class. This object contains dimensions of the model,
the observed data, and the model parameters. Once a \code{mixedMemModel} object is created,
the specified model can be fit for the data using the \code{mmVarFit} function. If the inputs are inconsistent (ie if dimensions do not match,
or if observations and distribution types are not compatible,\code{mixedMemModel} will throw an error. For additional details on usage, and model
assumptions, see the corresponding vignette "Fitting Mixed Membership Models Using \code{mixedMem}".
}
\examples{
set.seed(123)
Total <- 50 # 50 Individuals
J <- 3 # 3 different variables
# distributions of each variable
dist <- c("multinomial", "bernoulli", "rank")
# 100 repeated measures of the multinomial, 5 repeated measures of the
# Bernoulli, 1 repeated measure of the rank
Rj <- c(100, 5, 1)

K <- 4 # 4 sub-populations
alpha <- rep(.5, K) #hyperparameter for dirichlet distribution

# Number of categories/alternatives for each variable. For the Bernoulli, Vj = 1
Vj <- c(10, 1, 4)


theta <- array(0, dim = c(J, K, max(Vj)))
# Parameters governing multinomial
theta[1,,] <- gtools::rdirichlet(K, rep(.3, Vj[1]))
#parameters governing Bernoulli
theta[2,,] <- cbind(rbeta(K, 1,1), matrix(0, nrow = K, ncol = Vj[1]-1))
theta[3,,] <- cbind(gtools::rdirichlet(K, rep(.3, Vj[3])),
 matrix(0, nrow = K, ncol = Vj[1]-Vj[3]))

# Items selected for each observation. For Multinomial and Bernoulli, this is always 1
# For rank data, this will be the number of alternatives ranked
Nijr = array(0, dim = c(Total, J, max(Rj)))
Nijr[,1,c(1:Rj[1])] = 1 # N_ijr is 1 for multinomial variables
Nijr[,2,c(1:Rj[2])] = 1 # N_ijr is 1 for Bernoulli variables
Nijr[,3,c(1:Rj[3])] = sample.int(Vj[3], size = Total, replace = TRUE)

# generate random sample of observations
sampleMixedMem <- rmixedMem(Total, J, Rj, Nijr, K, Vj,
dist, theta, alpha)

## Initialize a mixedMemModel object
test_model <- mixedMemModel(Total = Total, J = J,Rj = Rj,
 Nijr= Nijr, K = K, Vj = Vj,dist = dist, obs = sampleMixedMem$obs,
  alpha = alpha, theta = theta)
# Look at Summary of the initialized model
summary(test_model)
# Plot the current values for theta
plot(test_model)
}

