% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/neModel.R
\name{neModel}
\alias{neModel}
\title{Fit a natural effect model}
\usage{
neModel(formula, family = gaussian, expData, xFit, se = c("bootstrap",
  "robust"), nBoot = 1000, parallel = c("no", "multicore", "snow"),
  ncpus = getOption("boot.ncpus", 1L), progress = TRUE, ...)
}
\arguments{
\item{formula}{a \code{\link[stats]{formula}} object providing a symbolic description of the natural effect model.}

\item{family}{a description of the error distribution and link
    function to be used in the model.  For \code{glm} this can be a
    character string naming a family function, a family function or the
    result of a call to a family function.  For \code{glm.fit} only the
    third option is supported.  (See \code{\link{family}} for details of
    family functions.)}

\item{expData}{the expanded dataset (of class \code{"\link{expData}"}).}

\item{xFit}{fitted model object representing a model for the exposure (used for inverse treatment (exposure) probability weighting).}

\item{se}{character string indicating the type of standard errors to be calculated. The default type is based on the bootstrap (see details).}

\item{nBoot}{number of bootstrap replicates (see \code{R} argument of \code{\link[boot]{boot}}).}

\item{parallel}{(only for bootstrap) The type of parallel operation to be used (if any). If missing, the default is taken from the option \code{"boot.parallel"} (and if that is not set, \code{"no"}).}

\item{ncpus}{(only for bootstrap) integer: number of processes to be used in parallel operation: typically one would chose this to the number of available CPUs (see details).}

\item{progress}{(only for bootstrap) logical value indicating whether or not a progress bar should be displayed. Progress bars are automatically disabled for multicore processing.}

\item{...}{additional arguments (passed to \code{\link[stats]{glm}}).}
}
\value{
An object of class \code{"\link[=neModel-methods]{neModel}"} (which additionally inherits from class \code{"neModelBoot"} if the bootstrap is used) consisting of a list of 3 objects:
\item{\code{neModelFit}}{the fitted natural model object (of class \code{"\link[stats]{glm}"}) with downwardly biased standard errors}
\item{\code{bootRes}, \code{vcov}}{the bootstrap results (of class \code{"\link[boot]{boot}"}; if \code{se = "bootstrap"}) or the robust variance-covariance matrix (if \code{se = "robust"})}
\item{\code{terms}}{the \code{neTerms} (internal class) object used. This object is equivalent to the \code{\link[=terms.object]{terms}} object returned by the \code{\link[stats]{glm}} function, 
but has an additional \code{"vartype"} attribute, a list including pointers to the names of the outcome variable (\code{Y}), exposure (\code{X}), mediator (\code{M}), covariates (\code{C}) and auxiliary hypothetical variables \emph{x} and \emph{x*} (\code{Xexp}).}
See \code{\link{neModel-methods}} for methods for \code{neModel} objects.
}
\description{
\code{neModel} is used to fit a natural effect model on the expanded dataset.
}
\details{
This function is a wrapper for \code{\link[stats]{glm}}, providing unbiased bootstrap (\code{se = "bootstrap"}, the default) or robust (\code{se = "robust"}) standard errors for the parameter estimates (see below for more details).

The \code{formula} argument requires to be specified in function of the variables from the expanded dataset (specified in \code{expData}) whose corresponding parameters index the direct and indirect effect.
Stratum-specific natural effects can be estimated by additionally modeling the relation between the outcome and baseline covariates.
If the set of baseline covariates adjusted for in the \code{formula} argument is not sufficient to control for confounding (e.g. when fitting a population-average natural effect model),
an adequate model for the exposure (conditioning on a sufficient set of baseline covariates) should be specified in the \code{xFit} argument.
In this case, such a model for the exposure distribution is needed to weight by the reciprocal of the probability (density) of the exposure (i.e. inverse probability weighting) in order to adjust for confounding.
Just as for ratio-of-mediator probability weighting (see paragraph below), this kind of weighting is done internally.

Quadratic or higher-order polynomial terms can be included in the \code{formula} by making use of the \code{\link[base]{I}} function or by using the \code{\link[stats]{poly}} function.
However, we do not recommend the use of orthogonal polynomials (i.e. using the default argument specification \code{raw = FALSE} in \code{poly}), as these are not compatible with the \code{\link[medflex]{neEffdecomp}} function.

In contrast to \code{\link[stats]{glm}}, the \code{expData} argument (rather than \code{data} argument) requires specification of a data frame that inherits from class \code{"\link{expData}"},
which contains additional information about e.g. the fitted working model, the variable types or terms of this working model 
and possibly ratio-of-mediator probability weights.
The latter are automatically extracted from the \code{\link{expData}} object and weighting is done internally.

As the default \code{\link[stats]{glm}} standard errors fail to reflect the uncertainty inherent to the working model(s) (i.e. either a model for the mediator or an imputation model for the outcome and possibly a model for the exposure),
bootstrap standard errors (using the \code{\link[boot]{boot}} function from the \pkg{boot} package) or robust standard errors are calculated. The default type of standard errors is bootstrap standard errors. 
Robust standard errors (based on the sandwich estimator) can be requested (to be calculated) instead by specifying \code{se = "robust"}.
}
\note{
It is important to note that the original mediator(s) should not be specified in the \code{formula} argument, as the natural indirect effect in natural effect models
should be captured solely by parameter(s) corresponding to the auxiliary hypothetical variable \emph{x*} in the expanded dataset (see \code{\link{expData}}).
}
\section{Bootstrap standard errors}{

The bootstrap procedure entails refitting all working models on each bootstrap sample, reconstructing the expanded dataset and subsequently refitting the specified natural effect model on this dataset.
In order to obtain stable standard errors, the number of bootstrap samples (specified via the \code{nBoot} argument) should be chosen relatively high (default is 1000).

To speed up the bootstrap procedure, parallel processing can be used by specifying the desired type of parallel operation via the \code{parallel} argument (for more details, see \code{\link[boot]{boot}}).
The number of parallel processes (\code{ncpus}) is suggested to be specified explicitly (its default is 1, unless the global option \code{options("boot.cpus")} is specified). 
The function \code{\link[parallel]{detectCores}} from the \pkg{parallel} package can be helpful at determining the number of available cores (although this may not always correspond to the number of \emph{allowed} cores).
}

\section{Robust standard errors}{

Robust variance-covariance matrices for the model parameters, based on the sandwich estimator, are calculated using core functions from the \pkg{sandwich} package.
Additional details and derivations for the sandwich estimator for natural effect models can be found in the corresponding vignette that can be obtained by the command \code{vignette("sandwich", package = "medflex")}.
}

\examples{
data(UPBdata)

##############################
## weighting-based approach ##
##############################
weightData <- neWeight(negaff ~ att + gender + educ + age, 
                       data = UPBdata)

## stratum-specific natural effects
# bootstrap SE
\dontrun{
weightFit1b <- neModel(UPB ~ att0 * att1 + gender + educ + age, 
                       family = binomial, expData = weightData)
summary(weightFit1b)
}
# robust SE
weightFit1r <- neModel(UPB ~ att0 * att1 + gender + educ + age, 
                       family = binomial, expData = weightData, se = "robust")
summary(weightFit1r)

## population-average natural effects
expFit <- glm(att ~ gender + educ + age, data = UPBdata)
# bootstrap SE
\dontrun{
weightFit2b <- neModel(UPB ~ att0 * att1, family = binomial, 
                       expData = weightData, xFit = expFit)
summary(weightFit2b)
}
# robust SE
weightFit2r <- neModel(UPB ~ att0 * att1, family = binomial, 
                       expData = weightData, xFit = expFit, se = "robust")
summary(weightFit2r)

###############################
## imputation-based approach ##
###############################
impData <- neImpute(UPB ~ att * negaff + gender + educ + age, 
                    family = binomial, data = UPBdata)

## stratum-specific natural effects
# bootstrap SE
\dontrun{
impFit1b <- neModel(UPB ~ att0 * att1 + gender + educ + age, 
                    family = binomial, expData = impData)
summary(impFit1b)
}
# robust SE
impFit1r <- neModel(UPB ~ att0 * att1 + gender + educ + age, 
                    family = binomial, expData = impData, se = "robust")
summary(impFit1r)

## population-average natural effects
# bootstrap SE
\dontrun{
impFit2b <- neModel(UPB ~ att0 * att1, family = binomial, 
                    expData = impData, xFit = expFit)
summary(impFit2b)
}
# robust SE
impFit2r <- neModel(UPB ~ att0 * att1, family = binomial, 
                    expData = impData, xFit = expFit, se = "robust")
summary(impFit2r)

\dontshow{# check with vgam (VGAM package)
# library(VGAM)
# weightData <- neWeight(negaff ~ att + gender + educ + age, family = "gaussianff", data = UPBdata, FUN = vgam)
# impData <- neImpute(UPB ~ att + negaff + gender + educ + age, family = "binomialff", data = UPBdata, FUN = vgam)
# debug(neModel)
# weightFit <- neModel(UPB ~ att0 + att1 + gender + educ + age, family = binomial, expData = weightData, nBoot = 2)
# impFit <- neModel(UPB ~ att0 + att1 + gender + educ + age, family = binomial, expData = impData, nBoot = 2)
# summary(weightFit)
# summary(impFit)

# warning!
impFit <- neModel(UPB ~ att0 * att1 + gender + educ + age, family = binomial, expData = impData, nBoot = 2)

# inverse propensity score weighting
expFit <- glm(att ~ gender + educ + age, data = UPBdata)
impFit <- neModel(UPB ~ att0 + att1, family = binomial, expData = impData, xFit = expFit, nBoot = 2)
}
}
\references{
Lange, T., Vansteelandt, S., & Bekaert, M. (2012). A Simple Unified Approach for Estimating Natural Direct and Indirect Effects. \emph{American Journal of Epidemiology}, \bold{176}(3), 190-195.

Vansteelandt, S., Bekaert, M., & Lange, T. (2012). Imputation Strategies for the Estimation of Natural Direct and Indirect Effects. \emph{Epidemiologic Methods}, \bold{1}(1), Article 7.

Loeys, T., Moerkerke, B., De Smet, O., Buysse, A., Steen, J., & Vansteelandt, S. (2013). Flexible Mediation Analysis in the Presence of Nonlinear Relations: Beyond the Mediation Formula. \emph{Multivariate Behavioral Research}, \bold{48}(6), 871-894.
}
\seealso{
\code{\link{neModel-methods}}, \code{\link{plot.neModel}}, \code{\link{neImpute}}, \code{\link{neWeight}}, \code{\link{neLht}}, \code{\link{neEffdecomp}}
}
