% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shorter_interval.R
\name{shorter_interval}
\alias{shorter_interval}
\alias{longer_interval}
\alias{shorter_duration}
\alias{longer_duration}
\title{Find the shorter or longer interval between two hours}
\usage{
shorter_interval(x, y)

longer_interval(x, y)

shorter_duration(x, y)

longer_duration(x, y)
}
\arguments{
\item{x, y}{An \code{\link[hms:hms]{hms}} or \code{\link[base:DateTimeClasses]{POSIXt}}
object.}
}
\value{
\itemize{
\item For \code{shorter_interval()} or \code{longer_interval()}, an
\code{\link[lubridate:interval]{Interval}} object with the shorter or longer
interval between \code{x} and \code{y}.
\item For \code{shorter_duration()} or \code{longer_duration()}, a
\code{\link[lubridate:duration]{Duration}} object with the shorter or longer
duration between \code{x} and \code{y}.
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#maturing}{\figure{lifecycle-maturing.svg}{options: alt='[Maturing]'}}}{\strong{[Maturing]}}

\code{shorter_interval()} returns the shorter interval between two
\code{\link[hms:hms]{hms}} or \code{\link[base:DateTimeClasses]{POSIXt}} object hours.

\code{longer_interval()} do the inverse of \code{shorter_interval()}, i.e., returns the
longer interval between two hours.

\code{shorter_duration()} and \code{longer_duration()} return the interval time span
of \code{shorter_interval()} and \code{longer_interval()} as
\code{\link[lubridate:duration]{Duration}} objects.
}
\details{
\subsection{The two intervals problem}{

Given two hours, \code{x} and \code{y}, in a two-day timeline, without date references,
there will be always two possible intervals between them, as illustrated
below.

To figure out what interval is the  shorter or the longer,
\code{shorter_interval()} and \code{longer_interval()} verify two scenarios: 1. When
\code{x} comes before \code{y}; and 2. when \code{x} comes after \code{y}. This only works if \code{x}
value is smaller than \code{y}, therefore, the function will make sure to swap \code{x}
and \code{y} values if the latter assumption is not true.

Because \code{shorter_interval()} objective is to find the shorter interval, if
\code{x} and \code{y} are equal, the shorter interval will have a length of 0 hours,
resulting in an interval from \code{x} to \code{x}. But, if \code{longer_interval()} is used
instead, the latter condition will return a interval with 24 hours of length
(from \code{x} to \code{x} + 1 day).

In cases when \code{x} and \code{y} distance themselves by 12 hours, there will be no
shorter or longer interval (they will have equal length). In these cases,
\code{shorter_interval()} and \code{longer_interval()} will return the same value
(an interval of 12 hours).\preformatted{             day 1                        day 2
     x                  y         x                  y
   06:00              22:00     06:00              22:00
-----|------------------|---------|------------------|----->
              16h           8h             16h
          longer int.  shorter int.   longer int.

              day 1                      day 2
     y                   x       y                   x
   13:00               08:00   13:00               08:00
-----|-------------------|-------|-------------------|----->
              19h           5h            19h
          longer int.  shorter int.  longer int.

    x,y             x,y             x,y             x,y
     x               y               x               y
   10:00           10:00           10:00           10:00
-----|---------------|---------------|---------------|----->
    0h              0h              0h              0h
            24h             24h             24h

              day 1                      day 2
     y               x               y               x
   12:00           00:00           12:00           00:00
-----|---------------|---------------|---------------|----->
            12h             12h             12h
}
}

\subsection{Class requirements}{

The \code{mctq} package works with a set of object classes specially created to
hold time values. These classes can be found in the \link[hms:hms-package]{hms}
and \link[lubridate:lubridate-package]{lubridate} package.
}

\subsection{Base date and timezone}{

\code{shorter_interval()} and \code{longer_interval()} use the
\href{https://en.wikipedia.org/wiki/Unix_time}{Unix epoch} (1970-01-01) date as
the start date for creating intervals.

The output will always have \code{"UTC"} set as timezone. Learn more about
time zones in \code{\link[base:timezones]{?timezone}}.
}

\subsection{\code{POSIXt} objects}{

\code{\link[base:DateTimeClasses]{POSIXt}} objects passed as argument to \code{x} or \code{y}
will be stripped of their dates. Only the time will be considered.

Both \code{\link[base:as.POSIXlt]{POSIXct}} and \code{\link[base:as.POSIXlt]{POSIXlt}} are
objects that inherits the class \code{\link[base:DateTimeClasses]{POSIXt}}. Learn more
about it in \code{\link[base:DateTimeClasses]{?DateTimeClasses}}.
}

\subsection{\code{NA} values}{

\code{shorter_interval()} or \code{longer_interval()} will return an
\code{\link[lubridate:interval]{Interval}} \code{NA}-\code{NA} if \code{x} or \code{y} are \code{NA}.

\code{shorter_duration()} or \code{longer_duration()} will return a
\code{\link[lubridate:duration]{Duration}} \code{NA}  if \code{x} or \code{y} are \code{NA}.
}
}
\examples{
## Scalar example

x <- hms::parse_hm("23:00")
y <- hms::parse_hm("01:00")

shorter_interval(x, y)
#> [1] 1970-01-01 23:00:00 UTC--1970-01-02 01:00:00 UTC # Expected
shorter_duration(x, y)
#> [1] "7200s (~2 hours)" # Expected
longer_interval(x, y)
#> [1] 1970-01-01 01:00:00 UTC--1970-01-01 23:00:00 UTC # Expected
longer_duration(x, y)
#> [1] "79200s (~22 hours)" # Expected

x <- lubridate::as_datetime("1985-01-15 12:00:00")
y <- lubridate::as_datetime("2020-09-10 12:00:00")

shorter_interval(x, y)
#> [1] 1970-01-01 12:00:00 UTC--1970-01-01 12:00:00 UTC # Expected
shorter_duration(x, y)
#> [1] "0s" # Expected
longer_interval(x, y)
#> [1] 1970-01-01 12:00:00 UTC--1970-01-02 12:00:00 UTC # Expected
longer_duration(x, y)
#> [1] "86400s (~1 days)" # Expected

## Vector example

x <- c(hms::parse_hm("15:30"), hms::parse_hm("21:30"))
y <- c(hms::parse_hm("19:30"), hms::parse_hm("04:00"))

shorter_interval(x, y)
#> [1] 1970-01-01 15:30:00 UTC--1970-01-01 19:30:00 UTC # Expected
#> [2] 1970-01-01 21:30:00 UTC--1970-01-02 04:00:00 UTC # Expected
shorter_duration(x, y)
#> [1] [1] "14400s (~4 hours)"   "23400s (~6.5 hours)" # Expected
longer_interval(x, y)
#> [1] 1970-01-01 19:30:00 UTC--1970-01-02 15:30:00 UTC # Expected
#> [2] 1970-01-01 04:00:00 UTC--1970-01-01 21:30:00 UTC # Expected
longer_duration(x, y)
#> [1] "72000s (~20 hours)"   "63000s (~17.5 hours)" # Expected
}
\seealso{
Other utility functions: 
\code{\link{assign_date}()},
\code{\link{cycle_time}()},
\code{\link{pretty_mctq}()},
\code{\link{qplot_walk}()},
\code{\link{random_mctq}()},
\code{\link{raw_data}()},
\code{\link{round_time}()},
\code{\link{sum_time}()}
}
\concept{utility functions}
