% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/krige.R
\name{krige.mcgf}
\alias{krige.mcgf}
\title{Obtain kriging forecasts for an \code{mcgf} object.}
\usage{
\method{krige}{mcgf}(
  x,
  newdata,
  model = c("all", "base", "empirical"),
  interval = FALSE,
  level = 0.95,
  ...
)
}
\arguments{
\item{x}{An \code{mcgf} object.}

\item{newdata}{A data.frame with the same column names as \code{x}. If \code{newdata}
is missing the forecasts at the original data points are returned.}

\item{model}{Which model to use. One of \code{all}, \code{base}, and \code{empirical}.}

\item{interval}{Logical; if TRUE, prediction intervals are computed.}

\item{level}{A numeric scalar between 0 and 1 giving the confidence level for
the intervals (if any) to be calculated. Used when \code{interval = TRUE}}

\item{...}{Additional arguments. Give \code{lag} and \code{horizon} if they are not
defined in \code{x} for the \code{empirical} model.}
}
\value{
A list of kriging forecasts (and prediction intervals).
}
\description{
Obtain kriging forecasts for an \code{mcgf} object.
}
\details{
It produces simple kriging forecasts for a zero-mean mcgf. It supports
kriging for the \code{empirical} model, the \code{base} model, and the \code{all} model
which is the general stationary model with the base and Lagrangian model
from \code{x}.

When \code{interval = TRUE}, confidence interval for each forecasts and each
horizon is given. Note that it does not compute confidence regions.
}
\examples{
data(sim1)
sim1_mcgf <- mcgf(sim1$data, dists = sim1$dists)
sim1_mcgf <- add_acfs(sim1_mcgf, lag_max = 5)
sim1_mcgf <- add_ccfs(sim1_mcgf, lag_max = 5)

# Fit a separable model and store it to 'sim1_mcgf'
fit_sep <- fit_base(
    sim1_mcgf,
    model = "sep",
    lag = 5,
    par_init = c(
        c = 0.001,
        gamma = 0.5,
        a = 0.3,
        alpha = 0.5
    ),
    par_fixed = c(nugget = 0)
)
sim1_mcgf <- add_base(sim1_mcgf, fit_base = fit_sep)

# Fit a Lagrangian model
fit_lagr <- fit_lagr(
    sim1_mcgf,
    model = "lagr_tri",
    par_init = c(v1 = 300, v2 = 300, lambda = 0.15),
    par_fixed = c(k = 2)
)

# Store the fitted Lagrangian model to 'sim1_mcgf'
sim1_mcgf <- add_lagr(sim1_mcgf, fit_lagr = fit_lagr)

# Calculate the simple kriging predictions and intervals
sim1_krige <- krige(sim1_mcgf, interval = TRUE)

# Calculate RMSE for each location
rmse <- sqrt(colMeans((sim1_mcgf - sim1_krige$fit)^2, na.rm = TRUE))
rmse

# Calculate MAE for each location
mae <- colMeans(abs(sim1_mcgf - sim1_krige$fit), na.rm = TRUE)
mae

# Calculate POPI for each location
popi <- colMeans(
    sim1_mcgf < sim1_krige$lower | sim1_mcgf > sim1_krige$upper,
    na.rm = TRUE
)
popi
}
\seealso{
Other functions on fitting an mcgf: 
\code{\link{add_base.mcgf}()},
\code{\link{add_lagr.mcgf}()},
\code{\link{fit_base.mcgf}()},
\code{\link{fit_lagr.mcgf}()}
}
\concept{functions on fitting an mcgf}
