% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimator.R
\name{estimator}
\alias{estimator}
\alias{estimator.cfaLoadings}
\alias{estimator.efaLoadings}
\alias{estimator.ols}
\alias{estimator.plscLoadings}
\alias{estimator.plsreg}
\alias{estimator.tsls}
\title{Parameter estimation of a model matrix}
\usage{
estimator.ols(S, modelMatrix, W, ..., C = NULL, IC = NULL, n = NULL)

estimator.tsls(S, modelMatrix, W, ..., C)

estimator.plscLoadings(S, modelMatrix, W, ...)

estimator.efaLoadings(S, modelMatrix, W, ..., fm = "minres")

estimator.cfaLoadings(S, modelMatrix, W, ...)

estimator.plsreg(S, modelMatrix, W, ..., C)
}
\arguments{
\item{S}{Covariance matrix of the data.}

\item{modelMatrix}{A model matrix with dependent variables on rows, independent variables on colums, and
non-zero elements indicating relationships. Can be either \code{inner}, \code{reflective},
or \code{formative} matrix.}

\item{W}{Weight matrix, where the indicators are on colums and composites are on the rows.}

\item{...}{All other arguments are either ignored or passed through to third party estimation functions.}

\item{C}{Correlation matrix of the composites.}

\item{IC}{Correlation matrix of the composites and indicators.}

\item{n}{sample size, used for calculating standard errors.}

\item{fm}{factoring method for estimating the factor loadings. Passed through to \code{\link[psych]{fa}}.}
}
\value{
A matrix with estimated parameters or a list of two matrices containing estimates and
standard errors.
}
\description{
Estimates the parameters of a model matrix (\code{inner},
\code{reflective}, or \code{formative}).
}
\details{
Parameter estimation functions estimate the parameters of a model matrix (\code{inner},
\code{reflective}, or \code{formative}). These functions can be used as \code{parametersInner},
\code{parametersReflective}, and \code{parametersFormative} arguments for 
\code{\link{parameterEstim.separate}}.

When two-stage least squares regression is applied with \code{estimator.tsls}, all
exogenous variables are used as instrumental varaables. There is currently no check of whether the
number of instrumental variables is sufficient for estimation.

\code{estimator.plscLoadings} estimates the loadings by scaling the weights \code{W} with the
correction factor \code{c} presented by Dijkstra (2011). This produces a MINRES estimator,
which constraints the loadings to be proportional to the weights.
The PLSc code is loosely based on code contributed by Wenjing Huang and developed with the guidance
by Theo Dijkstra.

\code{estimator.plscLoadings} estimates loadings with an unconstrained single factor model,
which requires at least three indicators per block. The loadings of 
single indicator factors are estimated as 1 and two indicator factors as estimated by the
square root of the indicator correlation.

Providing \code{C} or \code{IC} allows for using disattenuated or otherwise
adjusted correlation matrices. If not provided, these matrices are calculated using \code{S} and
\code{W}.

A part of the code for \code{\link{estimator.plsreg}} is adopted from the \pkg{plspm} package, licenced
under GPL3.
}
\section{Functions}{
\itemize{
\item \code{estimator.ols}: parameter estimation with OLS regression. Can be applied to \code{inner}, \code{reflective},
or \code{formative} matrix.

\item \code{estimator.tsls}: parameter estimation with two-stage least squares regression. For \code{inner} matrix only.

\item \code{estimator.plscLoadings}: parameter estimation with Dijkstra's (2011) PLSc correction for loadings.  For \code{reflective} matrix only.

\item \code{estimator.efaLoadings}: parameter estimation with one indicator block at at time with exploratory
factor analysis using the \code{\link[psych]{fa}} function from the \code{psych} package. For \code{reflective} matrix only.

\item \code{estimator.cfaLoadings}: Estimates a maximum likelihood confirmatory factor analysis with \code{\link[lavaan]{lavaan}}.  For \code{reflective} matrix only.

\item \code{estimator.plsreg}: parameter estimation with PLS regression. For \code{inner} matrix only.
}}
\examples{
# Run the education example from the book

# Sanchez, G. (2013) PLS Path Modeling with R
# Trowchez Editions. Berkeley, 2013. 
# http://www.gastonsanchez.com/PLS Path Modeling with R.pdf

education <- read.csv("http://www.gastonsanchez.com/education.csv")

Support <- c(0, 0, 0, 0, 0, 0)
Advising <- c(0, 0, 0, 0, 0, 0)
Tutoring <- c(0, 0, 0, 0, 0, 0)
Value <- c(1, 1, 1, 0, 0, 0)
# Omit two paths (compared to teh model in the book) to achieve 
# identification of the 2SLS analysis
Satisfaction <- c(0, 0, 1, 1, 0, 0)
Loyalty <- c(0, 0, 0, 0, 1, 0)

inner <- rbind(Support, Advising, Tutoring, Value, Satisfaction, Loyalty)


reflective <- diag(6)[c(rep(1,4),
                        rep(2,4),
                        rep(3,4),
                        rep(4,4),
                        rep(5,3),
                        rep(6,4)),]
formative <- matrix(0, 6, 23)

colnames(inner) <- colnames(reflective) <- rownames(formative) <- rownames(inner)
rownames(reflective) <- colnames(formative) <- colnames(education)[2:24]

education.model <- list(inner = inner,
              reflective = reflective,
              formative = formative)

# Reverse code two variables
education[,c("sup.under","loy.asha")] <- - education[,c("sup.under","loy.asha")]

S <- cor(education[,2:24])

# PLSc with OLS regression

education.out <- matrixpls(S,education.model,
                      disattenuate = TRUE,
                      parametersReflective = estimator.plscLoadings)

# PLSc with 2SLS regresssion

education.out2 <- matrixpls(S,education.model,
                      disattenuate = TRUE,
                      parametersReflective = estimator.plscLoadings,
                      parametersInner = estimator.tsls)


# Disattenuated regression with unit-weighted scales and exploratory factor analysis
# reliability estimates (with unconstrained MINRES estimator)

education.out3 <- matrixpls(S,education.model,
                       disattenuate = TRUE,
                       weightFun = weightFun.fixed,
                       parametersReflective = estimator.efaLoadings)

# Disattenuated GSCA with 2SLS regression after disattenuated based on 
# confirmatory factor analysis reliability estimates


education.out4 <- matrixpls(S,education.model,
                       disattenuate = TRUE,
                       innerEstim = innerEstim.gsca,
                       outerEstim = outerEstim.gsca,
                       parametersInner = estimator.tsls,
                       parametersReflective = estimator.cfaLoadings)


# Compare the results

cbind(PLSc = education.out, PLSc_2sls = education.out2, 
      DR = education.out3, GSCAc = education.out4)

# Compare the reliability estimates

cbind(PLSc = attr(education.out,"Q"), PLSc_2sls = attr(education.out2,"Q"), 
      DR = attr(education.out3,"Q"), GSCAc = attr(education.out4,"Q"))
}
\author{
Mikko Rönkkö, Wenjing Huang, Theo Dijkstra

Mikko Rönkkö, Gaston Sanchez, Laura Trinchera, Giorgio Russolillo
}
\references{
Huang, W. (2013). PLSe: Efficient Estimators and Tests for Partial Least Squares (Doctoral dissertation). University of California, Los Angeles.

Dijkstra, T. K. (2011). Consistent Partial Least Squares estimators for linear and polynomial factor models. A report of a belated, serious and not even unsuccessful attempt. Comments are invited. Retrieved from http://www.rug.nl/staff/t.k.dijkstra/consistent-pls-estimators.pdf

Sanchez, G. (2013). \emph{PLS Path Modeling with R.} Retrieved from http://www.gastonsanchez.com/PLS Path Modeling with R.pdf
}

