% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predictions.R
\name{predictions}
\alias{predictions}
\title{Adjusted Predictions}
\usage{
predictions(
  model,
  variables = NULL,
  newdata = NULL,
  conf.level = 0.95,
  type = "response",
  ...
)
}
\arguments{
\item{model}{Model object}

\item{variables}{Character vector. Compute Adjusted Predictions for
combinations of each of these variables. Factor levels are considered at
each of their levels. Numeric variables variables are considered at Tukey's
Five-Number Summaries. \code{NULL} uses the original data used to fit the model.}

\item{newdata}{A dataset over which to compute adjusted predictions. \code{NULL} uses
the original data used to fit the model.}

\item{conf.level}{The confidence level to use for the confidence interval.
No interval is computed if \code{conf.int=NULL}.  Must be strictly greater than 0
and less than 1. Defaults to 0.95, which corresponds to a 95 percent
confidence interval.}

\item{type}{Type(s) of prediction as string or vector This can
differ based on the model type, but will typically be a string such as:
"response", "link", "probs", or "zero".}

\item{...}{Additional arguments are pushed forward to \code{predict()}.}
}
\value{
A \code{data.frame} with one row per observation and several columns:
\itemize{
\item \code{rowid}: row number of the \code{newdata} data frame
\item \code{type}: prediction type, as defined by the \code{type} argument
\item \code{group}: (optional) value of the grouped outcome (e.g., categorical outcome models)
\item \code{predicted}: predicted outcome
\item \code{std.error}: standard errors computed by the \code{insight::get_predicted} function or, if unavailable, via \code{marginaleffects} delta method functionality.
\item \code{conf.low}: lower bound of the confidence or highest density interval (for bayesian models)
\item \code{conf.high}: upper bound of the confidence or highest density interval (for bayesian models)
}
}
\description{
Calculate adjusted predictions for each row of the dataset. The \code{datagrid()}
function and the \code{newdata} argument can be used to calculate Average
Adjusted Predictions (AAP), Average Predictions at the Mean (APM), or
Predictions at User-Specified Values of the regressors (aka Adjusted
Predictions at Representative values, APR). See the Details and Examples
sections below.
}
\details{
An "adjusted prediction" is the outcome predicted by a model for some
combination of the regressors' values, such as their observed values, their
means, or factor levels (a.k.a. “reference grid”).
When possible, this function uses the delta method to compute the standard
error associated with the adjusted predictions.

A detailed vignette on adjusted predictions is published on the package
website:

https://vincentarelbundock.github.io/marginaleffects/
Compute model-adjusted predictions (fitted values) for a "grid" of regressor values.
}
\examples{
# Adjusted Prediction for every row of the original dataset
mod <- lm(mpg ~ hp + factor(cyl), data = mtcars)
pred <- predictions(mod)
head(pred)

# Adjusted Predictions at User-Specified Values of the Regressors
predictions(mod, newdata = datagrid(hp = c(100, 120), cyl = 4))

# Average Adjusted Predictions (AAP)
library(dplyr)
mod <- lm(mpg ~ hp * am * vs, mtcars)

pred <- predictions(mod, newdata = datagrid(am = 0, grid.type = "counterfactual")) \%>\%
    summarize(across(c(predicted, std.error), mean))

predictions(mod, newdata = datagrid(am = 0:1, grid.type = "counterfactual")) \%>\% 
    group_by(am) \%>\%
    summarize(across(c(predicted, std.error), mean))

# Conditional Adjusted Predictions
plot_cap(mod, condition = "hp")
}
