#' Get `sf` polygon and lines of rivers, channels and other wetlands of Spain
#'
#' @description
#' Loads a `sf` polygon or line object representing rivers, channels,
#' reservoirs and other wetlands of Spain
#'
#' @family natural
#'
#' @return A `sf` polygon or line object.
#'
#'
#' @source IGN data via a custom CDN (see
#' <https://github.com/rOpenSpain/mapSpain/tree/sianedata>).
#'
#' @export
#'
#' @param resolution Resolution of the polygon. Values available are
#'   `"3"`, `"6.5"` or `"10"`.
#'
#' @inheritParams esp_get_hypsobath
#'
#' @param name Optional. A character or  [`regex`][base::grep()] expression
#' with the name of the element(s) to be extracted.
#'
#' @details
#' Metadata available on
#' <https://github.com/rOpenSpain/mapSpain/tree/sianedata/>.
#'
#' @examples
#' \donttest{
#' # Use of regex
#'
#' regex1 <- esp_get_rivers(name = "Tajo|Segura")
#' unique(regex1$rotulo)
#'
#'
#' regex2 <- esp_get_rivers(name = "Tajo$| Segura")
#' unique(regex2$rotulo)
#'
#' # See the diference
#'
#' # Rivers in Spain
#' shapeEsp <- esp_get_country(moveCAN = FALSE)
#'
#' MainRivers <-
#'   esp_get_rivers(name = "Tajo$|Ebro$|Ebre$|Duero|Guadiana$|Guadalquivir")
#'
#'
#' library(tmap)
#'
#' tm_shape(shapeEsp, bbox = MainRivers) +
#'   tm_borders() +
#'   tm_shape(MainRivers) +
#'   tm_lines(col = "skyblue", lwd = 3)
#'
#'
#' # Wetlands in South-West Andalucia
#' and <- esp_get_prov(c("Huelva", "Sevilla", "Cadiz"))
#' Wetlands <- esp_get_rivers(spatialtype = "area")
#'
#'
#' tm_shape(and) +
#'   tm_polygons() +
#'   tm_shape(Wetlands) +
#'   tm_polygons(col = "skyblue", alpha = 0.5, border.col = "skyblue", lwd = 2)
#' }
esp_get_rivers <- function(epsg = "4258",
                           cache = TRUE,
                           update_cache = FALSE,
                           cache_dir = NULL,
                           verbose = FALSE,
                           resolution = "3",
                           spatialtype = "line",
                           name = NULL) {
  # Check epsg
  init_epsg <- as.character(epsg)
  if (!init_epsg %in% c("4326", "4258", "3035", "3857")) {
    stop("epsg value not valid. It should be one of 4326, 4258, 3035 or 3857")
  }

  # Valid spatialtype
  validspatialtype <- c("area", "line")

  if (!spatialtype %in% validspatialtype) {
    stop(
      "spatialtype should be one of '",
      paste0(validspatialtype, collapse = "', "),
      "'"
    )
  }

  type <- paste0("river", spatialtype)

  # Get shape
  rivers_sf <-
    esp_hlp_get_siane(
      type,
      resolution,
      cache,
      cache_dir,
      update_cache,
      verbose,
      Sys.Date()
    )

  # Get river names
  rivernames <-
    esp_hlp_get_siane(
      "rivernames",
      resolution,
      cache,
      cache_dir,
      update_cache,
      verbose,
      Sys.Date()
    )



  # Merge names
  rivernames$id_rio <- rivernames$PFAFRIO
  rivernames <- rivernames[, c("id_rio", "NOM_RIO")]

  rivers_sf_merge <- merge(rivers_sf,
    rivernames,
    all.x = TRUE
  )

  if (!is.null(name)) {
    getrows1 <- grep(name, rivers_sf_merge$rotulo)
    getrows2 <- grep(name, rivers_sf_merge$NOM_RIO)
    getrows <- unique(c(getrows1, getrows2))
    rivers_sf_merge <- rivers_sf_merge[getrows, ]

    if (nrow(rivers_sf_merge) == 0) {
      stop(
        "Your value '",
        name,
        "' for name does not produce any result ",
        "for spatialtype = '",
        spatialtype,
        "'"
      )
    }
  }

  if (spatialtype == "area") {
    rivers_sf_merge <-
      rivers_sf_merge[, -match("NOM_RIO", colnames(rivers_sf_merge))]
  }

  rivers_sf_merge <-
    sf::st_transform(rivers_sf_merge, as.double(init_epsg))
  return(rivers_sf_merge)
}
