% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mable-ph-model.r
\name{mable.ph}
\alias{mable.ph}
\title{Mable fit of Cox's proportional hazards regression model}
\usage{
mable.ph(formula, data, M, g = NULL, pi0 = NULL, tau = Inf,
  x0 = NULL, controls = mable.ctrl(), progress = TRUE)
}
\arguments{
\item{formula}{regression formula. Response must be \code{cbind}.  See 'Details'.}

\item{data}{a dataset}

\item{M}{a positive integer or a vector \code{(m0, m1)}. If \code{M = m} or \code{m0 = m1 = m},
then \code{m} is a preselected degree. If \code{m0<m1} it specifies the set of 
consective candidate model degrees \code{m0:m1} for searching an optimal degree,
where \code{m1-m0>3}.}

\item{g}{initial guess of \eqn{d}-vector of regression coefficients.  See 'Details'.}

\item{pi0}{Initial guess of \eqn{\pi(x_0) = F(\tau_n|x_0)}. Without right censored data, \code{pi0 = 1}. See 'Details'.}

\item{tau}{right endpoint of support \eqn{[0, \tau)} must be greater than or equal to the maximum observed time}

\item{x0}{a working baseline covariate. See 'Details'.}

\item{controls}{Object of class \code{mable.ctrl()} specifying iteration limit 
and other control options. Default is \code{\link{mable.ctrl}}.}

\item{progress}{if \code{TRUE} a text progressbar is displayed}
}
\value{
A list with components
\itemize{ 
  \item \code{m} the selected/preselected optimal degree \code{m}
  \item \code{p} the estimate of \eqn{p = (p_0, \dots, p_m, p_{m+1})}, the coefficients of Bernstein polynomial of degree \code{m}
  \item \code{coefficients} the estimated regression coefficients of the PH model
  \item \code{SE} the standard errors of the estimated regression coefficients 
  \item \code{z} the z-scores of the estimated regression coefficients 
  \item \code{mloglik} the maximum log-likelihood at an optimal degree \code{m}
  \item \code{tau.n} maximum observed time \eqn{\tau_n}
  \item \code{tau} right endpoint of support \eqn{[0, \tau)}
  \item \code{x0} the working baseline covariates 
  \item \code{egx0} the value of \eqn{e^{\gamma'x_0}} 
  \item \code{convergence} an integer code, 1 indicates either the EM-like 
    iteration for finding maximum likelihood reached the maximum iteration for at least one \code{m} 
    or the search of an optimal degree using change-point method reached the maximum candidate degree,
    2 indicates both occured, and 0 indicates a successful completion.  
  \item \code{delta} the final \code{delta} if \code{m0 = m1} or the final \code{pval} of the change-point 
     for searching the optimal degree \code{m};
 }
 and, if \code{m0<m1},
\itemize{
  \item \code{M} the vector \code{(m0, m1)}, where \code{m1} is the last candidate degree when the search stoped
  \item \code{lk} log-likelihoods evaluated at \eqn{m \in \{m_0,\ldots, m_1\}}
  \item \code{lr} likelihood ratios for change-points evaluated at \eqn{m \in \{m_0+1, \ldots, m_1\}}
  \item \code{pval} the p-values of the change-point tests for choosing optimal model degree
  \item \code{chpts} the change-points chosen with the given candidate model degrees
}
}
\description{
Maximum approximate Bernstein/Beta likelihood estimation in Cox's proportional hazards regression model 
based on interal censored event time data.
}
\details{
Consider Cox's PH model with covariate for interval-censored failure time data: 
\eqn{S(t|x) = S(t|x_0)^{\exp(\gamma'(x-x_0))}}, where \eqn{x_0} satisfies \eqn{\gamma'(x-x_0)\ge 0}.   
  Let \eqn{f(t|x)} and \eqn{F(t|x) = 1-S(t|x)} be the density and cumulative distribution
functions of the event time given \eqn{X = x}, respectively.
Then \eqn{f(t|x_0)} on \eqn{[0, \tau_n]} can be approximated by  
\eqn{f_m(t|x_0, p) = \tau_n^{-1}\sum_{i=0}^m p_i\beta_{mi}(t/\tau_n)},
where \eqn{p_i \ge 0}, \eqn{i = 0, \ldots, m}, \eqn{\sum_{i=0}^mp_i = 1-p_{m+1}},  
\eqn{\beta_{mi}(u)} is the beta denity with shapes \eqn{i+1} and \eqn{m-i+1}, and 
\eqn{\tau_n} is the largest observed time, either uncensored time, or right endpoint of interval/left censored,
or left endpoint of right censored time. So we can approximate  \eqn{S(t|x_0)} on \eqn{[0, \tau_n]} by
\eqn{S_m(t|x_0; p) = \sum_{i=0}^{m+1} p_i \bar B_{mi}(t/\tau_n)}, where 
\eqn{\bar B_{mi}(u)}, \eqn{i = 0, \ldots, m}, is the beta survival function with shapes 
 \eqn{i+1} and \eqn{m-i+1}, \eqn{\bar B_{m,m+1}(t) = 1}, \eqn{p_{m+1} = 1-\pi(x_0)}, and
\eqn{\pi(x_0) = F(\tau_n|x_0)}. For data without right-censored time, \eqn{p_{m+1} = 1-\pi(x_0) = 0}.

Response variable should be of the form \code{cbind(l, u)}, where \code{(l, u)} is the interval 
containing the event time. Data is uncensored if \code{l = u}, right censored 
if \code{u = Inf} or \code{u = NA}, and  left censored data if \code{l = 0}.
The associated covariate contains \eqn{d} columns. The baseline \code{x0} should chosen so that 
\eqn{\gamma'(x-x_0)} is nonnegative for all the observed \eqn{x} and 
all \eqn{\gamma} in a neighborhood of its true value.

A missing initial value of \code{g} is imputed by \code{ic_sp()} of package \code{icenReg}. 

 The search for optimal degree \code{m} is stoped if either \code{m1} is reached or the test 
 for change-point results in a p-value \code{pval} smaller than \code{sig.level}.
This process takes longer than \code{\link{maple.ph}} to select an optimal degree.
}
\examples{
\donttest{
   # Ovarian Cancer Survival Data
   require(survival)
   futime2<-ovarian$futime
   futime2[ovarian$fustat==0]<-Inf
   ovarian2<-data.frame(age=ovarian$age, futime1=ovarian$futime, futime2=futime2)
   ova<-mable.ph(cbind(futime1, futime2) ~ age, data = ovarian2, M=c(2,35), g=.16)
   op<-par(mfrow=c(2,2))
   plot(ova, which = "likelihood")
   plot(ova, which = "change-point")
   plot(ova, y=data.frame(c(60)), which="survival", add=FALSE, type="l", 
         xlab="Days", main="Age = 60")
   plot(ova, y=data.frame(c(65)), which="survival", add=FALSE, type="l", 
         xlab="Days", main="Age = 65")
   par(op)
}
}
\references{
Guan, Z. (2019) Maximum Approximate Bernstein Likelihood Estimation in Proportional Hazard Model for Interval-Censored Data, 
arXiv:1906.08882 .
}
\seealso{
\code{\link{maple.ph}}
}
\author{
Zhong Guan <zguan@iusb.edu>
}
\concept{Cox proportional hazards model}
\concept{interval censoring}
\keyword{distribution}
\keyword{models}
\keyword{nonparametric}
\keyword{regression}
\keyword{smooth}
\keyword{survival}
